/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.web.jetty.client;

import java.net.CookieStore;
import java.net.HttpCookie;
import java.net.URI;
import org.appops.core.mime.MimeType;
import org.appops.web.common.client.Cookie;
import org.appops.web.common.client.Proxy;
import org.appops.web.common.client.ServerResponse;
import org.appops.web.common.client.WebClient;
import org.appops.web.common.client.WebClientException;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.HttpProxy;
import org.eclipse.jetty.client.Origin;
import org.eclipse.jetty.client.ProxyConfiguration;
import org.eclipse.jetty.client.api.ContentProvider;
import org.eclipse.jetty.client.api.ContentResponse;
import org.eclipse.jetty.client.api.Request;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.util.ssl.SslContextFactory;

/**
 * Wrapper class with wraps actual jetty http client. It also exposes methods to send get and post
 * requests.
 *
 * @author deba
 * @version $Id: $Id
 */
public class JettyWebClient extends WebClient<HttpClient> {



  /**
   * <p>Constructor for JettyWebClient.</p>
   */
  public JettyWebClient() {
    super();
    setWrappedClient(createHttpClient());
  }

  private static HttpClient createHttpClient() {
    SslContextFactory ssl = new SslContextFactory.Client();
    ssl.setEndpointIdentificationAlgorithm("HTTPS");
    HttpClient http = new HttpClient(ssl);
    return http;
  }

  /** {@inheritDoc} */
  @Override
  public ServerResponse get(String url) {
    return get(url, null);
  }

  /** {@inheritDoc} */
  @Override
  public ServerResponse get(String url, Cookie cookie) {
    try {
      start();
      Request request = wrappedClient().newRequest(url);
      request.method(HttpMethod.GET);
      if (cookie != null) {
        request.cookie(createHttpCookie(cookie));
      }
      ContentResponse contentResponse = request.send();
      return convertToServerResponse(contentResponse);
    } catch (Exception e) {
      throw new WebClientException(e);
    } finally {
      stop();
    }
  }


  /** {@inheritDoc} */
  @Override
  public ServerResponse post(String url, Object content, MimeType contentType) {
    return post(url, content, contentType, null);
  }

  /** {@inheritDoc} */
  @Override
  public ServerResponse post(String url, Object content, MimeType contentType, Cookie cookie) {
    try {
      start();
      Request request = wrappedClient().POST(url);
      if (cookie != null) {
        request.cookie(createHttpCookie(cookie));
      }
      if (content != null && contentType != null) {
        ContentProvider contentProvider = new ContentProviderCreator().create(content);
        request.content(contentProvider, contentType.value());
      }
      ContentResponse contentResponse = request.send();
      return convertToServerResponse(contentResponse);
    } catch (Exception e) {
      if (e instanceof WebClientException) {
        throw (WebClientException) e;
      } else {
        throw new WebClientException("URL- >" + url, e);
      }
    } finally {
      stop();
    }
  }

  /**
   * Converts jetty content response into appops response.
   * 
   * @param contentResponse Jetty content response.
   * @return Appops server response.
   */
  private ServerResponse convertToServerResponse(ContentResponse contentResponse) {
    ServerResponse response = new ServerResponse();
    response.setContent(contentResponse.getContent());
    response.setContentAsString(contentResponse.getContentAsString());
    response.setEncoding(contentResponse.getEncoding());
    response.setMediaType(contentResponse.getMediaType());
    response.setStatus(contentResponse.getStatus());;
    return response;
  }



  /**
   * Starts http client if it's not running.
   */
  void start() {
    try {
      if (!isRunning()) {
        wrappedClient().start();
      }
    } catch (Exception e) {
      throw new WebClientException(e);
    }
  }

  /**
   * Stops running http client.
   */
  void stop() {
    try {
      if (isRunning()) {
        wrappedClient().stop();
      }
    } catch (Exception e) {
      throw new WebClientException(e);
    }

  }

  /**
   * Checks if wrapped http client instance is running or not.
   * 
   * @return true, if client is running, false otherwise.
   */
  boolean isRunning() {
    return wrappedClient().isRunning();
  }


  /** {@inheritDoc} */
  @Override
  public void addCookie(String url, Cookie cookie) {
    CookieStore cookieStore = wrappedClient().getCookieStore();
    cookieStore.add(URI.create(url), createHttpCookie(cookie));
  }

  /** {@inheritDoc} */
  @Override
  public void addProxy(Proxy proxy) {
    ProxyConfiguration proxyConfig = wrappedClient().getProxyConfiguration();
    proxyConfig.getProxies().add(createHttpProxy(proxy));
  }

  /**
   * Create HttpProxy instance from apppops proxy passed.
   * 
   * @param proxy Appops proxy.
   * @return Http proxy instance generated.
   */
  private HttpProxy createHttpProxy(Proxy proxy) {
    HttpProxy httpProxy =
        new HttpProxy(new Origin.Address(proxy.host(), proxy.port()), proxy.isSecure());
    return httpProxy;
  }

  /**
   * Create HttpCookie instance from apppops cookie passed.
   * 
   * @param cookie Appops cookie.
   * @return Http cookie instance generated.
   */
  HttpCookie createHttpCookie(Cookie cookie) {
    HttpCookie httpCookie = new HttpCookie(cookie.name(), cookie.value());
    httpCookie.setDomain(cookie.domain());
    httpCookie.setHttpOnly(cookie.isHttpOnly());
    httpCookie.setMaxAge(cookie.maxAge());
    httpCookie.setSecure(cookie.isSecure());
    httpCookie.setVersion(cookie.version());
    httpCookie.setPath(cookie.path());
    return httpCookie;
  }



}
