/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */
package org.appops.web.jetty.client;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.appops.core.mime.MimeType;
import org.appops.web.common.client.AsyncWebClient;
import org.appops.web.common.client.Cookie;
import org.appops.web.common.client.Proxy;
import org.asynchttpclient.AsyncCompletionHandler;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.asynchttpclient.Dsl;
import org.asynchttpclient.ListenableFuture;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import org.asynchttpclient.cookie.CookieStore;
import org.eclipse.jetty.client.HttpProxy;
import org.eclipse.jetty.client.Origin;
import io.netty.handler.codec.http.cookie.DefaultCookie;

public class AsynHttpClient extends AsyncWebClient<AsyncHttpClient> {

  public AsynHttpClient() {
    AsyncHttpClient client = Dsl.asyncHttpClient();
    setWrappedAsyncHttpClient(client);
  }

  private static AsyncHttpClient createHttpClient() {

    DefaultAsyncHttpClientConfig.Builder clientBuilder = Dsl.config().setConnectTimeout(500);
    AsyncHttpClient client = Dsl.asyncHttpClient(clientBuilder);
    return client;
  }

  @Override
  public CompletableFuture<Response> get(String url) {
    return get(url, null);
  }

  @Override
  public CompletableFuture<Response> get(String url, Cookie cookie) {
    CompletableFuture<Response> response =
        wrappedAsyncHttpClient().prepareGet(url).execute().toCompletableFuture();

    return response;
  }

  @Override
  public CompletableFuture<Response> post(String url, Object content, MimeType contentType) {
    return post(url, content, contentType, null);
  }

  @Override
  public CompletableFuture<Response> post(String url, Object content, MimeType contentType,
      Cookie cookie) {
    CompletableFuture<Response> response = wrappedAsyncHttpClient().preparePost(url)
        .setBody(content.toString()).execute().toCompletableFuture();

    return response;
  }

  private ListenableFuture<Response> processAsyncRequest(Request request) {
    ListenableFuture<Response> listenableFuture =
        wrappedAsyncHttpClient().executeRequest(request, new AsyncCompletionHandler<Response>() {

          @Override
          public Response onCompleted(Response response) throws Exception {
            return response;
          }

        });
    return listenableFuture;
  }


  public CompletableFuture<String> helperFunction() {
    ExecutorService executor = Executors.newCachedThreadPool();

    final CompletableFuture<String> future = CompletableFuture.supplyAsync(() -> {
      return "42";
    }, executor);
    return future;
  }

  @Override
  public void addCookie(String url, Cookie cookie) {
    CookieStore cookieStore = wrappedAsyncHttpClient().getConfig().getCookieStore();
    DefaultCookie defaultCookie = createHttpCookie(cookie);
    cookieStore.add(org.asynchttpclient.uri.Uri.create(url), defaultCookie);
  }

  @Override
  public void addProxy(Proxy proxy) {}


  /**
   * Create HttpProxy instance from apppops proxy passed.
   * 
   * @param proxy Appops proxy.
   * @return Http proxy instance generated.
   */
  private HttpProxy createHttpProxy(Proxy proxy) {
    HttpProxy httpProxy =
        new HttpProxy(new Origin.Address(proxy.host(), proxy.port()), proxy.isSecure());
    return httpProxy;
  }

  /**
   * Create HttpCookie instance from apppops cookie passed.
   * 
   * @param cookie Appops cookie.
   * @return Http cookie instance generated.
   */
  DefaultCookie createHttpCookie(Cookie cookie) {
    DefaultCookie httpCookie = new DefaultCookie(cookie.name(), cookie.value());
    httpCookie.setDomain(cookie.domain());
    httpCookie.setHttpOnly(cookie.isHttpOnly());
    httpCookie.setMaxAge(cookie.maxAge());
    httpCookie.setSecure(cookie.isSecure());
    httpCookie.setPath(cookie.path());
    return httpCookie;
  }

}
