/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.web.jetty.client;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import org.appops.core.response.Blob;
import org.appops.core.service.Parameter;
import org.appops.web.common.client.MultiPartData;
import org.appops.web.common.client.WebClientException;
import org.eclipse.jetty.client.api.ContentProvider;
import org.eclipse.jetty.client.util.InputStreamContentProvider;
import org.eclipse.jetty.client.util.MultiPartContentProvider;
import org.eclipse.jetty.client.util.PathContentProvider;
import org.eclipse.jetty.client.util.StringContentProvider;

/**
 * Creates appropriate content provider class which wraps actual content which
 * is to be sent with jetty http post request.
 *
 * @author deba
 * @version $Id: $Id
 */
public class ContentProviderCreator {

	/**
	 * Creates an appropriate content provider passed for content by looking at type
	 * of the content specified.
	 * 
	 * @param content Actual data which is expected to be sent with request.
	 * 
	 * @return Content provider instance in which actual content is wrapped.
	 */
	ContentProvider create(Object content) {
		if (content == null) {
			throw new WebClientException("Cannot create ContentProvider for null content.");
		}
		try {
			if (content instanceof String) {
				return new StringContentProvider((String) content);
			} else if (content instanceof File) {
				return new PathContentProvider(Paths.get(((File) content).getPath()));
			} else if (content instanceof MultiPartData) {
				return createMultipartContentProvider((MultiPartData) content);
			} else {
				throw new WebClientException(
						"Content type -> " + content.getClass().getSimpleName() + " not supported yet.");
			}
		} catch (Exception e) {
			throw new WebClientException(e);
		}

	}

	/**
	 * Creates multipart content provider instance using data passed.
	 * 
	 * @param multipartData Instance containing multi part fields.
	 * @return Jetty multi part content provider.
	 * @throws IOException - if an I/O error occurs.
	 */
	private MultiPartContentProvider createMultipartContentProvider(MultiPartData multipartData) throws IOException {
		MultiPartContentProvider mulipartContentProvider = new MultiPartContentProvider();
		for (String fieldName : multipartData.fieldNames()) {
			Object value = multipartData.getField(fieldName);
			if (value instanceof Parameter && ((Parameter) value).getTypeName().equals(Blob.class.getCanonicalName())) {
				Blob blob = (Blob) ((Parameter) value).getValue();
				mulipartContentProvider.addFilePart(fieldName, blob.getNameOfFile(),
						new InputStreamContentProvider(blob.getInputStream(), blob.getInputStream().available()), null);
			} else {
				ContentProvider contentProvider = create(value);
				mulipartContentProvider.addFieldPart(fieldName, contentProvider, null);
			}
		}
		return mulipartContentProvider;
	}
}
