/*
 * AppOps is a Java framework to develop, deploy microservices with ease and is available for free
 * and common use developed by AinoSoft ( www.ainosoft.com )
 *
 * AppOps and AinoSoft are registered trademarks of Aino Softwares private limited, India.
 *
 * Copyright (C) <2016> <Aino Softwares private limited>
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU General Public License as published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version along with applicable additional terms as
 * provisioned by GPL 3.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License and applicable additional terms
 * along with this program.
 *
 * If not, see <https://www.gnu.org/licenses/> and <https://www.appops.org/license>
 */

package org.appops.web.jetty.server;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import java.net.URL;
import java.util.logging.Logger;
import org.appops.core.deployment.JettyConfig;
import org.appops.web.common.server.WebServer;
import org.appops.web.common.server.WebServerException;
import org.eclipse.jetty.annotations.AnnotationConfiguration;
import org.eclipse.jetty.plus.webapp.EnvConfiguration;
import org.eclipse.jetty.plus.webapp.PlusConfiguration;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.session.DatabaseAdaptor;
import org.eclipse.jetty.server.session.DefaultSessionCache;
import org.eclipse.jetty.server.session.DefaultSessionIdManager;
import org.eclipse.jetty.server.session.JDBCSessionDataStoreFactory;
import org.eclipse.jetty.server.session.SessionCache;
import org.eclipse.jetty.server.session.SessionHandler;
import org.eclipse.jetty.util.resource.JarResource;
import org.eclipse.jetty.webapp.Configuration;
import org.eclipse.jetty.webapp.FragmentConfiguration;
import org.eclipse.jetty.webapp.JettyWebXmlConfiguration;
import org.eclipse.jetty.webapp.MetaInfConfiguration;
import org.eclipse.jetty.webapp.WebAppContext;
import org.eclipse.jetty.webapp.WebInfConfiguration;
import org.eclipse.jetty.webapp.WebXmlConfiguration;

/**
 * A wrapper class which wraps actual jetty server instance.
 *
 * @author deba
 * @version $Id: $Id
 */
public class JettyServer extends WebServer<Server> {

  private Logger logger = Logger.getLogger(this.getClass().getCanonicalName());

  /**
   * <p>
   * Constructor for JettyServer.
   * </p>
   *
   * @param port a int.
   */
  public JettyServer(int port) {
    super(port, new Server(port));
  }


  /** {@inheritDoc} */
  @Override
  public JettyServer connect() {
    try {
      wrappedServer().start();
    } catch (Exception e) {
      throw new WebServerException(e);
    }
    return this;
  }

  /** {@inheritDoc} */
  @Override
  public void disconnect() {
    try {
      wrappedServer().stop();
    } catch (Exception e) {
      throw new WebServerException(e);
    }
  }



  /** {@inheritDoc} */
  @Override
  public boolean isConnected() {
    return wrappedServer().isRunning();
  }

  /** {@inheritDoc} */
  @Override
  public void configureAppContext(String contextPath, URL webappDirOrWarPath, String serviceName,
      JettyConfig jettyConfig) {
    WebAppContext context = new WebAppContext();
    context.setConfigurations(
        new Configuration[] {new AnnotationConfiguration(), new WebInfConfiguration(),
            new WebXmlConfiguration(), new MetaInfConfiguration(), new FragmentConfiguration(),
            new EnvConfiguration(), new PlusConfiguration(), new JettyWebXmlConfiguration()});
    context.setAttribute("org.eclipse.jetty.server.webapp.ContainerIncludeJarPattern", ".*");
    context.setContextPath(contextPath);
    context.setAttribute("org.eclipse.jetty.server.webapp.ContainerIncludeJarPattern",
        ".*/[^/]*jstl.*\\.jar$");
    context.setAttribute("org.eclipse.jetty.server.Request.setAsyncSupported", true);

    if (webappDirOrWarPath.toString().endsWith(".war")) {
      context.setWar(webappDirOrWarPath.toString());
    } else {
      context.setBaseResource(JarResource.newResource(webappDirOrWarPath));
    }
    configureDefaultSessionIdManager(context, serviceName);
    addSqlSessionHandler(context, jettyConfig);
    wrappedServer().insertHandler(context);
  }

  /**
   * It configured the defult session id manager.
   *
   * @param context instance of {@link org.eclipse.jetty.webapp.WebAppContext}
   * @param serviceName service name
   */
  public void configureDefaultSessionIdManager(WebAppContext context, String serviceName) {
    Server server = this.wrappedServer();
    DefaultSessionIdManager defaultSessionIdManager = new DefaultSessionIdManager(server);
    defaultSessionIdManager.setWorkerName(serviceName);
    server.setSessionIdManager(defaultSessionIdManager);
  }

  /**
   * It add the SQL session handler to given {@link WebAppContext}.
   * 
   * @param context instance of {@link WebAppContext}
   * @param jettyConfig database configuration for jetty session
   */

  void addSqlSessionHandler(WebAppContext context, JettyConfig jettyConfig) {

    SessionHandler sessionHandler = context.getSessionHandler();
    SessionCache sessionCache = new DefaultSessionCache(sessionHandler);
    sessionCache
        .setSessionDataStore(getJdbcDataStoreFactory(jettyConfig.getDriver(), jettyConfig.getUrl(),
            jettyConfig.getUser(), jettyConfig.getPassword()).getSessionDataStore(sessionHandler));
    sessionHandler.setSessionCache(sessionCache);
    sessionHandler.setHttpOnly(true);
  }

  /**
   * It return configured JDBC data store factory instance.
   * 
   * @param driver driver info
   * @param url database schema url
   * @param user username
   * @param password user password
   * @return instance of {@link JDBCSessionDataStoreFactory}
   */
  JDBCSessionDataStoreFactory getJdbcDataStoreFactory(String driver, String url, String user,
      String password) {

    HikariConfig hikariConfig = new HikariConfig();
    hikariConfig.setJdbcUrl(url);
    hikariConfig.addDataSourceProperty("url", url);
    hikariConfig.addDataSourceProperty("user", user);
    hikariConfig.addDataSourceProperty("password", password);
    HikariDataSource datasources = new HikariDataSource(hikariConfig);

    DatabaseAdaptor databaseAdaptor = new DatabaseAdaptor();
    databaseAdaptor.setDriverInfo(driver, url);
    databaseAdaptor.setDatasource(datasources);
    JDBCSessionDataStoreFactory jdbcSessionDataStoreFactory = new JDBCSessionDataStoreFactory();
    jdbcSessionDataStoreFactory.setDatabaseAdaptor(databaseAdaptor);
    return jdbcSessionDataStoreFactory;
  }

}
