
/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.atmosphere.core;

import com.sun.jersey.api.model.AbstractMethod;
import com.sun.jersey.spi.container.ContainerRequest;
import com.sun.jersey.spi.container.ContainerRequestFilter;
import com.sun.jersey.spi.container.ContainerResponse;
import com.sun.jersey.spi.container.ContainerResponseFilter;
import com.sun.jersey.spi.container.ContainerResponseWriter;
import com.sun.jersey.spi.container.ResourceFilter;
import com.sun.jersey.spi.container.ResourceFilterFactory;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.http.HttpServletRequest;
import org.atmosphere.core.annotation.Broadcast;
import org.atmosphere.core.annotation.Resume;
import org.atmosphere.core.annotation.Suspend;
import org.atmosphere.cpr.AtmosphereEvent;
import org.atmosphere.util.LoggerUtils;
import javax.ws.rs.core.Context;
import org.atmosphere.core.annotation.ResumeOnBroadcast;
import org.atmosphere.cpr.AtmosphereHandler;
import org.atmosphere.cpr.DefaultBroadcaster;
import org.atmosphere.handler.ReflectorServletProcessor;

/**
 * {@link ResourceFilterFactory} which intercept the response and appropriately
 * set the {@link AtmosphereEvent} filed based on the annotation the application
 * has defined.
 * 
 * @author Jeanfrancois Arcand
 */
public class AtmosphereFilter implements ResourceFilterFactory {

    private Logger logger = LoggerUtils.getLogger();

    static enum Action {SUSPEND, RESUME, BROADCAST, RESUME_ON_BROADCAST}
    
    private @Context HttpServletRequest httpServletRequest;

    private class Filter implements ResourceFilter,
            ContainerResponseFilter {

        private Action action;
        private int value;
        private Suspend.SCOPE scope;

        protected Filter(Action action, int value, Suspend.SCOPE scope) {
            this.action = action;
            this.value = value;
            this.scope = scope;
        }


        public ContainerRequestFilter getRequestFilter() {
            return null;
        }

        public ContainerResponseFilter getResponseFilter() {
            return this;
        }

        /**
         * Configure the {@link AtmosphereEvent} state (suspend, resume, broadcast)
         * based on the annotation the web application has used.
         *
         * @param request the {@link ContainerRequest}
         * @param response the {@link ContainerResponse}
         * @return the {@link ContainerResponse}
         */
        public ContainerResponse filter(ContainerRequest request, ContainerResponse response) {
            AtmosphereEvent event = (AtmosphereEvent) httpServletRequest.getAttribute(ReflectorServletProcessor.ATMOSPHERE_EVENT);
            AtmosphereHandler handler = (AtmosphereHandler) httpServletRequest.getAttribute(ReflectorServletProcessor.ATMOSPHERE_HANDLER);

            if (action == Action.SUSPEND) {
                Object o = response.getEntity();
                response.setContainerResponseWriter(
                        new Adapter(response.getContainerResponseWriter()));
                if (o != null) {
                    try {
                        response.getOutputStream().write(o.toString().getBytes());
                        response.getOutputStream().flush();
                    } catch (IOException ex) {
                        logger.log(Level.SEVERE, null, ex);
                    }
                }

                if (scope == Suspend.SCOPE.REQUEST) {
                    event.setBroadcaster(new DefaultBroadcaster());
                }

                event.suspend(value);
            } else if (action == Action.RESUME) {
                resume(event);
            } else if (action == Action.BROADCAST) {
                broadcast(response,event);
            } else if (action == Action.RESUME_ON_BROADCAST){
                broadcast(response,event);
                resume(event);
            }
            
            return response;
        }
    }

    void resume(AtmosphereEvent e){
        Iterator<AtmosphereEvent> i = e.getBroadcaster().getAtmosphereEvents();
        while(i.hasNext()){
            i.next().resume();
        }  
    }

    void broadcast(ContainerResponse r, AtmosphereEvent e){
        Object o = r.getEntity();

        // Something went wrong if null.
        if (o instanceof Broadcastable) {
            r.setEntity(((Broadcastable) o).broadcast());
        } else if (o != null) {
            e.getBroadcaster().broadcast(o);
        }
    }

    /**
     * Simple {@link ContainerResponseWriter} that prevent Jersey from
     * commiting the response.
     */
    private final class Adapter implements ContainerResponseWriter {

        private final ContainerResponseWriter crw;

        Adapter(ContainerResponseWriter crw) {
            this.crw = crw;
        }

        public OutputStream writeStatusAndHeaders(long contentLength,
                ContainerResponse response) throws IOException {
            return crw.writeStatusAndHeaders(contentLength, response);
        }

        public void finish() throws IOException {
        }
    }

    /**
     * Create a {@link ResourceFilter} which contains the information about the
     * annotation being processed.
     *
     * XXX Need to filter invalid mix of annotation.
     * 
     * @param am an {@link AbstractMethod}
     * @return a List of {@link ResourceFilter} to invoke.
     */
    public List<ResourceFilter> create(AbstractMethod am) {
        LinkedList<ResourceFilter> list = new LinkedList<ResourceFilter>();

        // We MUST make sure the Broadcast event is always processed first
        if (am.isAnnotationPresent(Broadcast.class)) {
            list.addLast((ResourceFilter)(ResourceFilter)
                    new Filter(Action.BROADCAST, 0,null));
        }

        if (am.isAnnotationPresent(Suspend.class)) {
            int value = am.getAnnotation(Suspend.class).value();
            Suspend.SCOPE scope = am.getAnnotation(Suspend.class).scope();
            list.addFirst((ResourceFilter)new Filter(Action.SUSPEND, value,scope ));
        }

        if (am.isAnnotationPresent(Resume.class)) {
            int value = am.getAnnotation(Resume.class).value();
            list.addFirst((ResourceFilter)(ResourceFilter)
                    new Filter(Action.RESUME, value, null));
        }

        if (am.isAnnotationPresent(ResumeOnBroadcast.class)) {
            list.addLast((ResourceFilter)(ResourceFilter)
                    new Filter(Action.RESUME_ON_BROADCAST, 0,null));
        }

        // Nothing, normal Jersey application.
        return list.size() > 0 ? list:null;
    }
}
