
/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.atmosphere.core;

import java.util.concurrent.ExecutionException;
import org.atmosphere.cpr.ClusterBroadcastFilter;
import com.sun.jersey.api.model.AbstractMethod;
import com.sun.jersey.spi.container.ContainerRequest;
import com.sun.jersey.spi.container.ContainerRequestFilter;
import com.sun.jersey.spi.container.ContainerResponse;
import com.sun.jersey.spi.container.ContainerResponseFilter;
import com.sun.jersey.spi.container.ContainerResponseWriter;
import com.sun.jersey.spi.container.ResourceFilter;
import com.sun.jersey.spi.container.ResourceFilterFactory;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.http.HttpServletRequest;
import org.atmosphere.core.annotation.Broadcast;
import org.atmosphere.core.annotation.Resume;
import org.atmosphere.core.annotation.Suspend;
import org.atmosphere.cpr.AtmosphereEvent;
import org.atmosphere.util.LoggerUtils;
import javax.ws.rs.core.Context;
import org.atmosphere.core.annotation.BroadcastFilter;
import org.atmosphere.core.annotation.Cluster;
import org.atmosphere.core.annotation.ResumeOnBroadcast;
import org.atmosphere.cpr.Broadcaster;
import org.atmosphere.cpr.BroadcasterConfig;
import org.atmosphere.handler.ReflectorServletProcessor;

/**
 * {@link ResourceFilterFactory} which intercept the response and appropriately
 * set the {@link AtmosphereEvent} filed based on the annotation the application
 * has defined.
 * 
 * @author Jeanfrancois Arcand
 */
public class AtmosphereFilter implements ResourceFilterFactory {

    private Logger logger = LoggerUtils.getLogger();

    final static String CONTAINER_RESPONSE = "cr";

    static enum Action {SUSPEND, RESUME, BROADCAST, RESUME_ON_BROADCAST, NONE}
    
    private @Context HttpServletRequest httpServletRequest;

    private class Filter implements ResourceFilter, ContainerResponseFilter {

        private ConcurrentLinkedQueue<Broadcaster> configuredBroadcaster
                = new ConcurrentLinkedQueue<Broadcaster>();

        private Action action = Action.NONE;
        private int value = 0;
        private Suspend.SCOPE scope;
        private Class<org.atmosphere.cpr.BroadcastFilter>[] filters = null;

        private final ArrayList<ClusterBroadcastFilter> clusters
                = new ArrayList<ClusterBroadcastFilter>();

        protected Filter(Action action) {
            this.action = action;
        }

        protected Filter(Action action, int value) {
            this.action = action;
            this.value = value;
        }

        protected Filter(Action action, int value, Suspend.SCOPE scope) {
            this.action = action;
            this.value = value;
            this.scope = scope;
        }


        public ContainerRequestFilter getRequestFilter() {
            return null;
        }

        public ContainerResponseFilter getResponseFilter() {
            return this;
        }

        /**
         * Configure the {@link AtmosphereEvent} state (suspend, resume, broadcast)
         * based on the annotation the web application has used.
         *
         * @param request the {@link ContainerRequest}
         * @param response the {@link ContainerResponse}
         * @return the {@link ContainerResponse}
         */
        public ContainerResponse filter(ContainerRequest request, ContainerResponse response) {
            AtmosphereEvent event = (AtmosphereEvent) httpServletRequest
                    .getAttribute(ReflectorServletProcessor.ATMOSPHERE_EVENT);
 
            if (action == Action.SUSPEND) {
                Object o = response.getEntity();
                response.setContainerResponseWriter(
                        new Adapter(response.getContainerResponseWriter()));
                if (o != null) {
                    try {
                        response.write();
                    } catch (IOException ex) {
                        logger.log(Level.SEVERE, null, ex);
                    }
                }

                Broadcaster bc = event.getBroadcaster();
                if (scope == Suspend.SCOPE.REQUEST) {
                    try {
                        bc = bc.getClass().newInstance();
                        event.setBroadcaster(bc);
                    } catch (InstantiationException ex) {
                        logger.log(Level.SEVERE, null, ex);
                    } catch (IllegalAccessException ex) {
                        logger.log(Level.SEVERE, null, ex);
                    }
                }
                
                httpServletRequest.getSession().setAttribute("suspendedEvent", event);
                httpServletRequest.getSession().setAttribute(CONTAINER_RESPONSE, response);
                event.suspend(value);
            } else if (action == Action.RESUME) {
                resume(event);
            } else if (action == Action.BROADCAST) {
                AtmosphereEvent e = (AtmosphereEvent)httpServletRequest.getAttribute("suspendedEvent");
                if (e != null){
                    event = e;
                }
                broadcast(response,event);
            } else if (action == Action.RESUME_ON_BROADCAST){
                broadcast(response,event);
                resume(event);
            }
            
            return response;
        }


        void configureFilter(Broadcaster bc){
            BroadcasterConfig c = bc.getBroadcasterConfig();

            // Always the first one, before any transformation/filtering
            for (ClusterBroadcastFilter cbf: clusters){
                cbf.setBroadcaster(bc);
                c.addFilter(cbf);
            }

            org.atmosphere.cpr.BroadcastFilter f = null;
            if (filters != null){
                for (Class<org.atmosphere.cpr.BroadcastFilter> filter: filters){
                    try{
                        f = filter.newInstance();
                    } catch (Throwable t){
                        logger.warning("Invalid @BroadcastFilter: " + filter);
                    }
                    c.addFilter(f);
                }
            }
        }

        private void setFilter(Class<org.atmosphere.cpr.BroadcastFilter>[] filters) {
            this.filters = filters;
        }


        void broadcast(ContainerResponse r, AtmosphereEvent e){
            Object o = r.getEntity();

            // Something went wrong if null.
            if (o instanceof Broadcastable) {
                addFilter(((Broadcastable<String>) o).b);
                r.setEntity(((Broadcastable<String>) o).broadcast());
            } else if (o != null) {
                addFilter(e.getBroadcaster());
                try {
                    e.getBroadcaster().broadcast(o).get();
                } catch (InterruptedException ex) {
                    logger.log(Level.SEVERE, null, ex);
                } catch (ExecutionException ex) {
                    logger.log(Level.SEVERE, null, ex);
                }
            }
        }

        void addFilter(Broadcaster bc){
            if (!configuredBroadcaster.contains(bc)){
                configureFilter(bc);
                configuredBroadcaster.offer(bc);
            }
        }
        
        void resume(AtmosphereEvent e){
            Iterator<AtmosphereEvent> i = e.getBroadcaster().getAtmosphereEvents();
            while(i.hasNext()){
                i.next().resume();
            }
        }

        void addCluster(ClusterBroadcastFilter f){
            clusters.add(f);
        }
    }


    /**
     * Simple {@link ContainerResponseWriter} that prevent Jersey from
     * commiting the response.
     */
    private final class Adapter implements ContainerResponseWriter {

        private final ContainerResponseWriter crw;

        Adapter(ContainerResponseWriter crw) {
            this.crw = crw;
        }

        public OutputStream writeStatusAndHeaders(long contentLength,
                ContainerResponse response) throws IOException {
            return crw.writeStatusAndHeaders(contentLength, response);
        }

        public void finish() throws IOException {
        }
    }

    /**
     * Create a {@link ResourceFilter} which contains the information about the
     * annotation being processed.
     *
     * XXX Need to filter invalid mix of annotation.
     * 
     * @param am an {@link AbstractMethod}
     * @return a List of {@link ResourceFilter} to invoke.
     */
    public List<ResourceFilter> create(AbstractMethod am) {
        LinkedList<ResourceFilter> list = new LinkedList<ResourceFilter>();

        if (am.isAnnotationPresent(Broadcast.class)) {

            Filter f = new Filter(Action.BROADCAST);
            list.addLast((ResourceFilter)(ResourceFilter)f);

            if (am.isAnnotationPresent(BroadcastFilter.class)){
                Class[] value = am.getAnnotation(BroadcastFilter.class).value();
                f.setFilter(value);
            }

            if (am.isAnnotationPresent(Cluster.class)){
                Class[] value = am.getAnnotation(Cluster.class).value();
                for (Class<ClusterBroadcastFilter> c: value){
                    try {
                        ClusterBroadcastFilter cbf = c.newInstance();
                        cbf.setClusterName(am.getAnnotation(Cluster.class).name());
                        f.addCluster(cbf);
                    } catch (Throwable t){
                        logger.log(Level.WARNING,"Invalid ClusterBroadcastFilter", t);
                    }
                }
            }
        }

        if (am.isAnnotationPresent(Suspend.class)) {
            int value = am.getAnnotation(Suspend.class).value();
            Suspend.SCOPE scope = am.getAnnotation(Suspend.class).scope();
            list.addFirst((ResourceFilter)new Filter(Action.SUSPEND, value,scope ));
        }

        if (am.isAnnotationPresent(Resume.class)) {
            int value = am.getAnnotation(Resume.class).value();
            list.addFirst((ResourceFilter)(ResourceFilter)
                    new Filter(Action.RESUME, value));
        }

        if (am.isAnnotationPresent(ResumeOnBroadcast.class)) {
            list.addLast((ResourceFilter)(ResourceFilter)
                    new Filter(Action.RESUME_ON_BROADCAST));
        }

        // Nothing, normal Jersey application.
        return list.size() > 0 ? list:null;
    }
}
