/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.core;

import com.sun.jersey.spi.container.ContainerResponse;
import java.io.IOException;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.atmosphere.cpr.AtmosphereEvent;
import org.atmosphere.cpr.Broadcaster;
import org.atmosphere.cpr.BroadcasterConfig;
import org.atmosphere.cpr.BroadcasterFuture;
import org.atmosphere.cpr.DefaultBroadcaster;
import org.atmosphere.util.LoggerUtils;

/**
 * Speacial {@link Broadcaster} that use the {@link ContainerResponse} uner the hood
 * to serialize the response.
 * 
 * @author Jeanfrancois Arcand
 */
public class JerseyBroadcaster implements Broadcaster<HttpServletRequest, HttpServletResponse,Object>{

    private final ConcurrentLinkedQueue<ContainerResponse> containerResponses =
            new ConcurrentLinkedQueue<ContainerResponse>();
    protected final ConcurrentLinkedQueue<AtmosphereEvent<HttpServletRequest, HttpServletResponse>> events =
            new ConcurrentLinkedQueue<AtmosphereEvent<HttpServletRequest, HttpServletResponse>>();
    private BroadcasterConfig<Object> bc;
    private final BlockingQueue<Entry> messages =
            new LinkedBlockingQueue<Entry>();
    private final AtomicBoolean started = new AtomicBoolean(false);
    private SCOPE scope = SCOPE.APPLICATION;
    private String name = DefaultBroadcaster.class.getSimpleName();

    public JerseyBroadcaster() {
        bc = new BroadcasterConfig<Object>();
    }

    /**
     * {@inheritDoc}
     */
    public void destroy() {
        bc.destroy();
    }

    /**
     * {@inheritDoc}
     */
    public Iterator<AtmosphereEvent<HttpServletRequest, HttpServletResponse>> getAtmosphereEvents() {
        return events.iterator();
    }

    /**
     * {@inheritDoc}
     */
    public void setScope(SCOPE scope) {
        this.scope = scope;
    }

    /**
     * {@inheritDoc}
     */
    public SCOPE getScope() {
        return scope;
    }

    /**
     * {@inheritDoc}
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * {@inheritDoc}
     */
    public String getName() {
        return name;
    }

    final private class Entry {

        Object message;
        Object eventsToPush;
        BroadcasterFuture f;

        Entry(Object message, Object eventsToPush,BroadcasterFuture f) {
            this.message = message;
            this.eventsToPush = eventsToPush;
            this.f = f;
        }
    }

    final void start() {
        if (!started.getAndSet(true)) {
            bc.getExecutorService().submit(new Runnable() {

                public void run() {
                    Entry msg = null;
                    try {
                        msg = messages.take();
                        // Leader/follower
                        bc.getExecutorService().submit(this);
                        if (msg.eventsToPush == null) {
                            for (ContainerResponse r : containerResponses) {
                                synchronized (r) {
                                    r.reset();
                                    r.setEntity(msg.message);
                                    broadcast(r);
                                }
                            }
                        }
                    } catch (InterruptedException ex) {
                        LoggerUtils.getLogger().log(Level.SEVERE, null, ex);
                    } finally {
                        if (msg != null){
                            msg.f.done();
                        }
                    }
                }
            });
        }
    }

    final private void broadcast(final ContainerResponse r) {
        try {
            r.write();
        } catch (IOException ex) {
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
        } catch (IllegalStateException ex) {
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
            containerResponses.remove(r);
        }
    }

    /**
     * {@inheritDoc}
     */
    public BroadcasterFuture<Object> broadcast(Object msg) {
        start();
        msg = bc.filter(msg);
        if (msg == null) {
            return null;
        }
        BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(msg);
        messages.offer(new Entry(msg, null,f));
        return f;
    }

    /**
     * {@inheritDoc}
     */
    public BroadcasterFuture<Object> broadcast(Object msg, AtmosphereEvent event) {
        start();
        msg = bc.filter(msg);
        if (msg == null) {
            return null;
        }
        BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(msg);
        messages.offer(new Entry(msg, event,f));
        return f;
    }

    /**
     * {@inheritDoc}
     */
    public BroadcasterFuture<Object> broadcast(Object msg, Set<AtmosphereEvent> subset) {
        start();
        msg = bc.filter(msg);
        if (msg == null) {
            return null;
        }
        BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(msg);
        messages.offer(new Entry(msg, subset,f));
        return f;
    }

    /**
     * {@inheritDoc}
     */
    public AtmosphereEvent<HttpServletRequest, HttpServletResponse>
            addAtmosphereEvent(AtmosphereEvent<HttpServletRequest, HttpServletResponse> event) {
        containerResponses.add((ContainerResponse)event.getRequest()
                .getSession().getAttribute(AtmosphereFilter.CONTAINER_RESPONSE));
        events.offer(event);
        return event;
    }

    /**
     * {@inheritDoc}
     */
    public AtmosphereEvent<HttpServletRequest, HttpServletResponse>
            removeAtmosphereEvent(AtmosphereEvent<HttpServletRequest, HttpServletResponse> event) {
        containerResponses.remove((ContainerResponse)event.getRequest()
                .getSession().getAttribute(AtmosphereFilter.CONTAINER_RESPONSE));
        events.remove(event);
        return event;
    }

    /**
     * Set the {@link BroadcasterConfig} instance.
     * @param bf
     */
    public void setBroadcasterConfig(BroadcasterConfig bc) {
        this.bc = bc;
    }

    /**
     * Return the current {@link BroadcasterConfig}
     * @return the current {@link BroadcasterConfig}
     */
    public BroadcasterConfig getBroadcasterConfig() {
        return bc;
    }
}
