/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain 
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.container;

import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.catalina.CometEvent;
import org.apache.catalina.CometEvent.EventType;
import org.atmosphere.cpr.AtmosphereServlet.Action;
import org.atmosphere.cpr.AtmosphereServlet.AtmosphereHandlerWrapper;
import org.atmosphere.cpr.CometSupport;
import org.atmosphere.cpr.WebContainer;

/** 
 * Comet Portable Runtime implementation on top of Tomcat AIO.
 *
 * @author Jeanfrancois Arcand
 */
public class TomcatCometSupport extends WebContainer implements CometSupport {

    public final static String COMET_EVENT = "CometEvent";

    private final static IllegalStateException unableToDetectComet
            = new IllegalStateException(unableToDetectComet());

    public TomcatCometSupport(Map<String,AtmosphereHandlerWrapper> atmosphereHandlers){
        super(atmosphereHandlers);
        setWebServerName(getContainerName() + "-" + getContainerDottedVersion());
    }

    /**
     * {@inheritDoc}
     */
    public String getContainerName() {
        return "Tomcat";
    }

    /**
     * {@inheritDoc}
     */
    public String getContainerDottedVersion() {
        return "6.0.x";
    }

    /**
     * Invoked by the Tomcat AIO when a Comet request gets detected.
     * 
     * @param req the {@link HttpServeltRequest}
     * @param res the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    public Action service(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {

        CometEvent event = (CometEvent) req.getAttribute(COMET_EVENT);

        // Comet is not enabled.
        if (event == null){
            throw unableToDetectComet;
        }
        Action action = null;
        // For now, we are just interested in CometEvent.REA 
        if (event.getEventType() == EventType.BEGIN) {
            action = suspended(req, res);
            if (action.type == Action.TYPE.SUSPEND) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Suspending " + res);
                }

                // Do nothing except setting the times out
                try{
                    if (action.timeout != -1) {
                        event.setTimeout((int) action.timeout);
                    } else {
                        event.setTimeout(Integer.MAX_VALUE);
                    }
                } catch (UnsupportedOperationException ex){
                    // Swallow s Tomcat APR isn't supporting time out
                    // TODO: Must implement the same functionality using a
                    // Scheduler
                }
            } else if (action.type == Action.TYPE.RESUME) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Resuming " + res);
                }
                // TODO: Not sure we need to close here.
                event.close();
            } else {
                event.close();
            }
        } else if (event.getEventType() == EventType.READ) {

        } else if (event.getEventType() == EventType.ERROR) {
            event.close();
        } else if (event.getEventType() == EventType.END) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Resuming " + res);
            }

            action = resumed(req, res);
            if (action.type == Action.TYPE.SUSPEND) {
                // Do nothing except setting the times out
                if (action.timeout != -1) {
                    event.setTimeout((int) action.timeout);
                } else {
                    event.setTimeout(Integer.MAX_VALUE);
                }
            } 
            event.close();
        } else if (event.getEventSubType() == CometEvent.EventSubType.TIMEOUT) {
            if (logger.isLoggable(Level.FINE)){
                logger.fine("Timing out " + res);
            }             
            action = timedout(req, res);
            event.close();
        }
        return action;
    }


    /**
     * Tomcat was unable to detect Atmosphere's CometProcessor implementation.
     * @return an error message describing how to fix the issue.
     */
    private static String unableToDetectComet(){
        StringBuilder sb = new StringBuilder();
        sb.append("Tomcat failed to detect this is a Comet application because context.xml ");
        sb.append( "is missing or the Http11NioProtocol Connector is not enabled.");
        sb.append("\nEnable it or if enabled add the following content under your META-INF/context.xml of your war file.");
        sb.append("\n<Context>");
        sb.append("\t<Loader delegate=\"true\"/>");
        sb.append("\t</Context>");
        return sb.toString();
    }
}
