/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.cpr;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.catalina.CometEvent;
import org.apache.catalina.CometProcessor;
import org.atmosphere.container.BlockingIOCometSupport;
import org.atmosphere.container.JettyCometSupport;
import org.atmosphere.container.GlassFishv2CometSupport;
import org.atmosphere.container.GlassFishv3CometSupport;
import org.atmosphere.container.GrizzlyCometSupport;
import org.atmosphere.container.Servlet30Support;
import org.atmosphere.container.TomcatCometSupport;
import org.atmosphere.container.WebLogicCometSupport;
import org.atmosphere.util.AtmosphereConfigReader;
import org.atmosphere.util.AtmosphereConfigReader.Property;
import org.atmosphere.util.IntrospectionUtils;
import org.atmosphere.util.LoggerUtils;

import weblogic.servlet.http.AbstractAsyncServlet;
import weblogic.servlet.http.RequestResponseKey;
/**
 * The {@link AtmosphereServlet} acts as a dispatcher for {@link AtmosphereHandler}
 * defined in META-INF/atmosphere.xml, or if atmosphere.xml is missing, all classes
 * that implements {@link AtmosphereHandler} will be discovered and mapped using
 * the class's name. This {@link Servlet} can be defined inside an application's
 * web.xml using the following:
 * <p><pre><code>
    &lt;servlet&gt;
        &lt;description&gt;AtmosphereServlet&lt;/description&gt;
        &lt;servlet-name&gt;AtmosphereServlet&lt;/servlet-name&gt;
        &lt;servlet-class&gt;org.atmosphere.cpr.AtmosphereServlet&lt;/servlet-class&gt;
        &lt;load-on-startup&gt;0 &lt;/load-on-startup&gt;
    &lt;/servlet&gt;
    &lt;servlet-mapping&gt;
         &lt;servlet-name&gt;AtmosphereServlet&lt;/servlet-name&gt;
         &lt;url-pattern&gt;/Atmosphere &lt;/url-pattern&gt;
    &lt;/servlet-mapping>
 * </code></pre></p>
 * 
 * @author Jeanfrancois Arcand
 */
public class AtmosphereServlet extends AbstractAsyncServlet implements CometProcessor{

    // If we detect Servlet 3.0, should we still use the default 
    // native Comet API.
    private boolean useNativeImplementation = false;
    
    public final static String SERVLET_30 = "javax.servlet.AsyncListener";
    public final static String GLASSFISH_V2 = 
            "com.sun.enterprise.web.connector.grizzly.comet.CometEngine";
    public final static String TOMCAT = 
            "org.apache.coyote.http11.Http11NioProcessor";
    public final static String JETTY = "org.mortbay.util.ajax.Continuation";
    public final static String GRIZZLY = "com.sun.grizzly.http.servlet.ServletAdapter";
    public final static String GLASSFISH_V3 = "com.sun.grizzly.comet.CometEngine";   
    public final static String WEBLOGIC = "weblogic.servlet.http.FutureResponseModel";

    public final static Logger logger = LoggerUtils.getLogger();

    private ArrayList<String> possibleAtmosphereHandlersCandidate = new ArrayList<String>();

   /**
     * The list of {@link AtmosphereHandler} and their associated mapping.
     */
    private Map<String,AtmosphereHandlerWrapper> atmosphereHandlers = new HashMap<String, AtmosphereHandlerWrapper>();

    // The WebServer we are running on.
    private CometSupport cometSupport;

    // Simple struc for storing Grizzly and its associated Broadcaster.
    public static class AtmosphereHandlerWrapper{

        public AtmosphereHandler atmosphereHandler;
        public Broadcaster broadcaster;

        public AtmosphereHandlerWrapper(AtmosphereHandler g){
            this.atmosphereHandler = g;
            this.broadcaster = new DefaultBroadcaster(atmosphereHandler);
        }
    }


    /**
     * Simple class/struck that hold the current state.
     */
    public static class Action{
        
        public enum TYPE {SUSPEND, RESUME, NONE, CANCEL}
        
        public long timeout = -1L;
        
        public TYPE type;
        
        public Action(){
            type = TYPE.NONE;
        }
        
        public Action(TYPE type){
            this.type = type;
        }
        
        public Action(TYPE type,long timeout){
            this.timeout = timeout;
            this.type = type;
        }
    }
    
    /**
     * Create an Atmosphere Servlet.
     */
    public AtmosphereServlet(){
        if (System.getProperty("org.atmosphere.useNative") != null){
            useNativeImplementation = true;
        }
    }
    
    /**
     * Load the {@link AtmosphereHandler} associated with this AtmosphereServlet.
     * @param sc the {@link ServletContext}
     */
    @Override
    public void init(ServletConfig sc) throws ServletException{
        super.init(sc);
        try{
            //TODO -> Add support for WEB-INF/lib/*.jar
            URL url = sc.getServletContext().getResource("/WEB-INF/classes/");
            URLClassLoader urlC = new URLClassLoader(new URL[] {url}, 
                    Thread.currentThread().getContextClassLoader());
            loadAtmosphereDotXml(sc.getServletContext().
                    getResourceAsStream("/META-INF/atmosphere.xml"), urlC);
            if (atmosphereHandlers.size() == 0){
                logger.warning("Missing META-INF/atmosphere.xml." +
                    " Will try to autodetect AtmosphereHandler");
                autoDetectAtmosphereHandlers(sc.getServletContext(), urlC);
            }
        } catch (Throwable t){
            throw new ServletException(t);
        }
        autoDetectContainer();
        
        cometSupport.init(sc);
    }

    /**
     * Load AtmosphereHandler defined under META-INF/atmosphere.xml
     */
    protected void loadAtmosphereDotXml(InputStream stream, URLClassLoader c) 
            throws IOException{

        if (stream == null){
            return;
        }

        AtmosphereConfigReader reader = new AtmosphereConfigReader(stream);

        Map<String, String> atmosphereHandlerNames = reader.getAtmosphereHandlers();
        Set<Entry<String,String>> entries = atmosphereHandlerNames.entrySet();
        for (Entry<String,String> entry: entries){
            AtmosphereHandler g;
            try {
                g = (AtmosphereHandler) c.loadClass(entry.getValue()).newInstance();
                logger.info("Sucessfully loaded " + g
                    + " mapped to context-path " + entry.getKey());
                atmosphereHandlers.put(entry.getKey(), new AtmosphereHandlerWrapper(g));
                for (Property p: reader.getProperty(entry.getValue())){
                    IntrospectionUtils.setAttribute(g,p.name,p.value);
                }
            } catch (Throwable t) {
                logger.warning("Unable to load AtmosphereHandler class: " + entry.getValue());
            }
        }
    }
    
    /**
     * Auto detect the underlying Servlet Container we are running on.
     */
    protected void autoDetectContainer(){
        try{
            if (useNativeImplementation){
                logger.info("Forcing the use of the container's" +
                        " native Comet API implemenation instead of Servlet 3.0");
                throw new ClassNotFoundException("Skip");
            }         
            Class.forName(SERVLET_30);
            cometSupport = new Servlet30Support(atmosphereHandlers);
        } catch (ClassNotFoundException e){
            try {
                // GlassFish v2
                Class.forName(GLASSFISH_V2);
                cometSupport = new GlassFishv2CometSupport(atmosphereHandlers);
            } catch (ClassNotFoundException ex) {
                try {
                    // GlassFish v2
                    Class.forName(JETTY);
                    cometSupport = new JettyCometSupport(atmosphereHandlers);
                } catch (ClassNotFoundException ex2) {
                    // Try Tomcat now
                    try{
                        Class.forName(TOMCAT);
                        cometSupport = new TomcatCometSupport(atmosphereHandlers);
                    } catch (ClassNotFoundException ex3) {
                        try{
                            Class.forName(GRIZZLY);
                            cometSupport = new GrizzlyCometSupport(atmosphereHandlers);
                        } catch (ClassNotFoundException ex4) {
                            try{
                                Class.forName(GLASSFISH_V3);
                                cometSupport = new GlassFishv3CometSupport(atmosphereHandlers);
                            } catch (ClassNotFoundException ex5){
                                try{
                                    Class.forName(WEBLOGIC);
                                    cometSupport = new WebLogicCometSupport(atmosphereHandlers);
                                } catch (ClassNotFoundException ex6){                                
                                    cometSupport = new BlockingIOCometSupport(atmosphereHandlers);
                                }
                            }
                        }
                    }
                }
            }
        }
        
        logger.info("Atmosphere Framework running under container "
                + cometSupport.getContainerName()
                + " version " + cometSupport.getContainerDottedVersion());
    }

    /**
     * Auto detect instance of {@link AtmosphereHandler} in case META-INF/atmosphere.xml
     * is missing.
     * @param sc {@link ServletContext}
     * @param c {@link URLClassLoaser} to load the class.
     */
    protected void autoDetectAtmosphereHandlers(ServletContext sc, URLClassLoader c)
            throws MalformedURLException, URISyntaxException{
        String webInf = "/WEB-INF/classes/";
        String s = sc.getRealPath(webInf);

        // Weblogic bug
        if (s == null){
            URL u = sc.getResource(webInf);
            s = u.getPath();
        }
        
        File f = new File(s);
        if (f.isDirectory()){
            getFiles(f);
            for (String className: possibleAtmosphereHandlersCandidate){
                try {
                    className = className.substring(className.indexOf(webInf)
                            + webInf.length(), className.lastIndexOf(".")).replace('/','.');
                    AtmosphereHandler g = (AtmosphereHandler) c.loadClass(className).newInstance();
                    logger.info("Sucessfully loaded " + g
                        + " mapped to context-path " + g.getClass().getSimpleName());
                    atmosphereHandlers.put("/" + g.getClass().getSimpleName(), new AtmosphereHandlerWrapper(g));
                } catch (Throwable t) {
                    logger.finest(className + " is not a AtmosphereHandler");
                }
            }
        }
    }

    /**
     * Get the list of possible candidate to load as {@link AtmosphereHandler}
     * @param f the real path {@link File{
     */
    void getFiles(File f){
        File[] files = f.listFiles();
        for (File test: files){
            System.out.println("test: " + test.getPath());
            if (test.isDirectory()){
                getFiles(test);
            } else {
                String clazz = test.getAbsolutePath();
                if (clazz.endsWith(".class")){
                    possibleAtmosphereHandlersCandidate.add(clazz);
                }
            }
        }
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doHead(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doOptions(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doTrace(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
  
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doDelete(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doPut(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }   
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doGet(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doPost(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        try{
            cometSupport.service(req, res);
        } catch (IllegalStateException ex){
            if (ex.getMessage() != null && ex.getMessage().startsWith("Tomcat failed")){
                logger.warning(ex.getMessage());
                logger.warning("Using the BlockingIOCometSupport.");
                cometSupport = new BlockingIOCometSupport(atmosphereHandlers);
                service(req,res);
            } else {
                logger.log(Level.SEVERE,"AtmosphereServlet exception", ex);
                throw ex;
            }
        }
    }

    /**
     * Hack to support Tomcat AIO like other WebServer. This method is invoked
     * by Tomcat when it detect a {@link Servlet} implements the interface
     * {@link CometProcessor} without invoking {@link Servlet#service}
     * 
     * @param cometEvent the {@link CometEvent}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    public void event(CometEvent cometEvent) throws IOException, ServletException {
        HttpServletRequest req = cometEvent.getHttpServletRequest();
        HttpServletResponse res = cometEvent.getHttpServletResponse();
        req.setAttribute(TomcatCometSupport.COMET_EVENT, cometEvent);
        cometSupport.service(req, res);
    }

    /**
     * Weblogic specific comet based implementation.
     * 
     * @param rrk
     * @return
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    protected boolean doRequest(RequestResponseKey rrk) throws IOException, ServletException {
        try{
            Action action = cometSupport.service(rrk.getRequest(), rrk.getResponse());
            if (action.type == Action.TYPE.SUSPEND){
                if (action.timeout == -1){
                        rrk.setTimeout(Integer.MAX_VALUE);
                } else {
                    rrk.setTimeout((int)action.timeout);
                }
            }
            return action.type == Action.TYPE.SUSPEND;
        } catch (IllegalStateException ex){
            logger.log(Level.SEVERE,"AtmosphereServlet.doRequest exception", ex);
            throw ex;
        }        
    }
    
    /**
     * Weblogic specific comet based implementation.
     * 
     * @param rrk
     * @return
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
   protected void doResponse(RequestResponseKey rrk, Object arg1) throws IOException, ServletException {
        try{
            Action action = cometSupport.service(rrk.getRequest(), rrk.getResponse());
            if (action.type == Action.TYPE.SUSPEND){
                if (action.timeout == -1){
                    rrk.setTimeout(Integer.MAX_VALUE);
                } else {
                    rrk.setTimeout((int)action.timeout);
                }
            }
        } catch (IllegalStateException ex){
            logger.log(Level.SEVERE,"AtmosphereServlet.doResponse exception", ex);
            throw ex;
        }     
    }
   
    /**
     * Weblogic specific comet based implementation.
     * 
     * @param rrk
     * @return
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    protected void doTimeout(RequestResponseKey rrk) throws IOException, ServletException {
        //TODO
    }


}
