/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.cpr;

import java.util.Set;
import java.util.concurrent.ExecutorService;

/**
 * A Broadcaster is responsible for delivering messages to its subscribed
 * {@link AtmosphereEvent}, which are representing a suspended response.
 * {@link AtmosphereEvent} can be added using {@link Broadcaster#addAtmosphereEvent},
 * so when {@link #broadcast(java.lang.Object)} execute,
 * {@link AtmosphereHandler#onMessage(org.atmosphere.cpr.AtmosphereEvent)} will
 * by invoked and the suspended connection will have a chance to write the
 * message available using {@link AtmosphereEvent#getMessage()}
 * <br>
 * A {@link Broadcaster}, by default, wil use the calling Thread to execute the
 * broadcast operation. To enable parallel execution, you can set an {@link ExecutorService}
 * which will be used to concurrently broadcast message.
 * 
 * @author Jeanfrancois Arcand
 */
public interface Broadcaster<E extends AtmosphereEvent> {

    /**
     * Broadcast the {@link Object} to all suspended response, e.g. invoke
     * {@link AtmosphereHandler#onMessage}.
     * 
     * @param o and {@link Object} to be shared.
     */
    public void broadcast(Object o);

    /**
     * Broadcast the {@link Object} to all suspended response, e.g. invoke
     * {@link AtmosphereHandler#onMessage} with an instance of {@link AtmosphereEvent}, representing
     * a set of {@link AtmosphereHandler}.
     *
     * @param o and {@link Object} to be shared.
     * @param event an {@link AtmosphereEvent}
     */
    public void broadcast(Object o, E event);

    /**
     * Broadcast the {@link Object} to all suspended response, e.g. invoke
     * {@link AtmosphereHandler#onMessage} with a {@link Set} of  {@link AtmosphereEvent},
     * representing a set of {@link AtmosphereHandler}.
     *
     * @param o and {@link Object} to be shared.
     * @param event an {@link AtmosphereEvent}
     */
    public void broadcast(Object o, Set<E> subset);

    /**
     * Add a {@link AtmosphereEvent} to the list of item to be notified when
     * the {@link Broadcaster#broadcast} is invoked.
     *
     * @param event an {@link AtmosphereEvent}
     * @return true if added, false if it was already added.
     */
    public boolean addAtmosphereEvent(E event);

    /**
     * Remove a {@link AtmosphereEvent} from the list of item to be notified when
     * the {@link Broadcaster#broadcast} is invoked.
     *
     * @param event an {@link AtmosphereEvent}
     * @return true if removed, false if it was already removed.
     */
    public boolean removeAtmosphereEvent(E event);

    /**
     * Set an {@link ExecutorService} which can be used to dispatch
     * {@link AtmosphereEvent}. Set it to null if broadcast must be executed
     * using the calling thread (this is the default).
     * @param executorService to be used when broadcasting.
     */
    public void setExecutorService(ExecutorService executorService);

    /**
     * Return the {@link ExecutorService} this {@link Broadcaster} support.
     * By defaul it return null, which means the calling thread is used when
     * invoking {@link Broadcaster#broadcast}
     * @return An ExecutorService.
     */
   public ExecutorService getExecutorService();
}
