/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.cpr;

import java.io.IOException;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.logging.Level;
import org.atmosphere.util.LoggerUtils;

/**
 * {@link Broadcaster} implementation.
 *
 * Broadcast messages to suspended reponse using the caller's Thread.
 * This basic {@link Broadcaster} isn't using any {@link ExecutorService} to
 * broadcast messages, so it is single threaded.
 * 
 * @author Jeanfrancois Arcand
 */
public class DefaultBroadcaster implements Broadcaster<AtmosphereEventImpl> {

    private ConcurrentLinkedQueue<AtmosphereEventImpl> events =
            new ConcurrentLinkedQueue<AtmosphereEventImpl>();
    private final AtmosphereHandler atmosphereHandler;

    private ExecutorService executorService;

    public DefaultBroadcaster(AtmosphereHandler atmosphereHandler) {
        this.atmosphereHandler = atmosphereHandler;
    }

    /**
     * {@inheritDoc}
     */
    public void broadcast(Object msg) {
        for (AtmosphereEventImpl event : events) {
            try {
                event.setMessage(msg);
                broadcast(event);
            } catch (IOException ex) {
                LoggerUtils.getLogger().log(Level.WARNING, "", ex);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    public void broadcast(Object msg, AtmosphereEventImpl event) {
        try {
            event.setMessage(msg);
            broadcast(event);
        } catch (IOException ex) {
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
        }
    }
    
    /**
     * {@inheritDoc}
     */
    public void broadcast(Object msg, Set<AtmosphereEventImpl> subset) {
        for (AtmosphereEventImpl event : subset) {
            try {
                event.setMessage(msg);
                broadcast(event);
            } catch (IOException ex) {
                LoggerUtils.getLogger().log(Level.WARNING, "", ex);
            }
        }
    }

    /**
     * Broadcast an {@link AtmosphereEvent} using the calling Thread or
     * using an {@link ExecutorService}
     * @param event  an {@link AtmosphereEvent}
     * @throws java.io.IOException
     */
    private void broadcast(final AtmosphereEvent event) throws IOException{
        if (executorService == null){
            synchronized (atmosphereHandler) {
                atmosphereHandler.onMessage(event);
            }
        } else {
            synchronized(atmosphereHandler){
                executorService.submit(new Runnable() {
                    public void run() {
                        try{
                            atmosphereHandler.onMessage(event);
                        } catch (IOException ex){
                            // AtmosphereHandler.onException?
                            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
                        }
                    }
                });
            }
        }
    }


    /**
     * {@inheritDoc}
     */
    public boolean addAtmosphereEvent(AtmosphereEventImpl event) {
        return events.offer(event);
    }
    
    /**
     * {@inheritDoc}
     */
    public boolean removeAtmosphereEvent(AtmosphereEventImpl event) {
        return events.remove(event);
    }

    /**
     * {@inheritDoc}
     */
    public void setExecutorService(ExecutorService executorService) {
        this.executorService = executorService;
    }

    /**
     * {@inheritDoc}
     */
    public ExecutorService getExecutorService() {
        return executorService;
    }

}
