/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.cpr;

import java.io.IOException;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.atmosphere.cpr.AtmosphereServlet.Action;
import org.atmosphere.cpr.AtmosphereServlet.AtmosphereHandlerWrapper;
import org.atmosphere.util.LoggerUtils;

/**
 * Base class which implement the semantics of supending and resuming of a 
 * Comet Request.
 * 
 * @author Jeanfrancois Arcand
 */
abstract public class WebContainer {

    private static final Action timedoutAction = new AtmosphereServlet.Action();

    protected Logger logger = LoggerUtils.getLogger();
    
    protected final Map<String,AtmosphereHandlerWrapper> atmosphereHandlers;
    
    protected final static String EVENT = "Context";

    private String webServerName = "";
    
    private ServletContext servletContext;
    
    public WebContainer(Map<String,AtmosphereHandlerWrapper> atmosphereHandlers){
        this.atmosphereHandlers = atmosphereHandlers;
    }

    /**
     * Init this {@link WebContainer}.
     * @param sc the {@link ServletContext}
     * @throws javax.servlet.ServletException
     */
    public void init(ServletConfig sc) throws ServletException{
        this.servletContext = sc.getServletContext();
    }
    
    /**
     * All proprietary Comet based {@link Servlet} must invoke the suspended
     * method when the first request comes in. The returned value, of type
     * {@link Atmosphere#Action}, tells the proprietary Comet {@link Servlet}
     * to suspended or not the current {@link HttpServletResponse}.
     * 
     * @param req the {@link HttpServletRequest}
     * @param res the {@link HttpServletResponse}
     * @return action the Action operation.
     */
    public Action suspended(HttpServletRequest req, HttpServletResponse res)
        throws IOException, ServletException{     
        if (logger.isLoggable(Level.FINE)){
            logger.fine("(suspend) invoked:\n HttpServletRequest: " + req 
                    + "\n HttpServletResponse: " + res);
        }
        return action(req, res);
    }


    /**
     * Invoke the {@link AtmosphereHandler#onEvent} method.
     * @param req the {@link HttpServletRequest}
     * @param res the {@link HttpServletResponse}
     * @return action the Action operation.
     */
    Action action(HttpServletRequest req, HttpServletResponse res)
               throws IOException,ServletException{
        AtmosphereEventImpl event
                = (AtmosphereEventImpl) req.getSession().getAttribute(EVENT);
        AtmosphereHandlerWrapper g = map(req.getServletPath());
        if (event == null || event.isSuspended()){
            event = new AtmosphereEventImpl(servletContext,webServerName,g.broadcaster,req, res);
            req.getSession().setAttribute(EVENT,event);
        } else {
            event.reset(req,res);
        }

        Action action = ((AtmosphereEventImpl)g.atmosphereHandler.onEvent(event)).action;
        if (action.type == Action.TYPE.SUSPEND){
            g.broadcaster.addAtmosphereEvent(event);
        } else if (action.type == Action.TYPE.RESUME){
            g.broadcaster.removeAtmosphereEvent(event);
        }
        return action;
    }

    /**
     * Return the {@link AtmosphereHandler} mapped to the passed servlet-path.
     *
     * @param servletPath the {@link HttpServletResponse}
     * @return the {@link AtmosphereHandler} mapped to the passed servlet-path.
     */
    AtmosphereHandlerWrapper map(String servletPath) throws ServletException{
        // TO DO -> This is not optimal at all
        AtmosphereHandlerWrapper g = atmosphereHandlers.get(servletPath);
        if (g == null){
            throw new ServletException("Invalid AtmosphereHandler");
        }
        return g;
    }
    
    /**
     * All proprietary Comet based {@link Servlet} must invoke the resume
     * method when the Atmosphere's application decide to resume the {@link HttpServletResponse}.
     * The returned value, of type
     * {@link Atmosphere#Action}, tells the proprietary Comet {@link Servlet}
     * to resume (again), suspended or do nothing with the current {@link HttpServletResponse}.
     *
     * @param req the {@link HttpServletRequest}
     * @param res the {@link HttpServletResponse}
     * @return action the Action operation.
    */
    public Action resumed(HttpServletRequest req, HttpServletResponse res)
        throws IOException, ServletException{
        if (logger.isLoggable(Level.FINE)){
            logger.fine("(resumed) invoked:\n HttpServletRequest: " + req 
                    + "\n HttpServletResponse: " + res);
        }   
        return action(req, res);
    }
    
    /**
     * All proprietary Comet based {@link Servlet} must invoke the timedout
     * method when the underlying WebServer time out the {@link HttpServletResponse}.
     * The returned value, of type
     * {@link Atmosphere#Action}, tells the proprietary Comet {@link Servlet}
     * to resume (again), suspended or do nothing with the current {@link HttpServletResponse}.
     *
     * @param req the {@link HttpServletRequest}
     * @param res the {@link HttpServletResponse}
     * @return action the Action operation.
     */
    public Action timedout(HttpServletRequest req, HttpServletResponse res)
        throws IOException, ServletException{
        
        HttpSession session = req.getSession(false);  
        if (session != null) {
            AtmosphereEvent<HttpServletRequest,HttpServletResponse> event
                    = (AtmosphereEvent) session.getAttribute(EVENT);
            if (event != null){
                map(req.getServletPath()).broadcaster.removeAtmosphereEvent(event);
            }
        }
        return timedoutAction;// ((AtmosphereEventImpl)atmosphereHandler.onEvent(event)).action;;
    }


    /**
     * Set the WebServer's name.
     * @param webServerName the WebServer's name.
     */
    protected void setWebServerName(String webServerName){
        this.webServerName = webServerName;
    }

}
