/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.cpr;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.catalina.CometEvent;
import org.apache.catalina.CometProcessor;
import org.atmosphere.container.BlockingIOCometSupport;
import org.atmosphere.container.JettyCometSupport;
import org.atmosphere.container.GlassFishv2CometSupport;
import org.atmosphere.container.GlassFishv3CometSupport;
import org.atmosphere.container.GrizzlyCometSupport;
import org.atmosphere.container.JBossWebCometSupport;
import org.atmosphere.container.Jetty7CometSupport;
import org.atmosphere.container.Servlet30Support;
import org.atmosphere.container.TomcatCometSupport;
import org.atmosphere.container.WebLogicCometSupport;
import org.atmosphere.handler.ReflectorServletProcessor;
import org.atmosphere.util.AtmosphereConfigReader;
import org.atmosphere.util.AtmosphereConfigReader.Property;
import org.atmosphere.util.BroadcasterLookup;
import org.atmosphere.util.IntrospectionUtils;
import org.atmosphere.util.LoggerUtils;

import weblogic.servlet.http.AbstractAsyncServlet;
import weblogic.servlet.http.RequestResponseKey;
import org.jboss.servlet.http.HttpEvent;
import org.jboss.servlet.http.HttpEventServlet;
/**
 * The {@link AtmosphereServlet} acts as a dispatcher for {@link AtmosphereHandler}
 * defined in META-INF/atmosphere.xml, or if atmosphere.xml is missing, all classes
 * that implements {@link AtmosphereHandler} will be discovered and mapped using
 * the class's name. This {@link Servlet} can be defined inside an application's
 * web.xml using the following:
 * <p><pre><code>
    &lt;servlet&gt;
        &lt;description&gt;AtmosphereServlet&lt;/description&gt;
        &lt;servlet-name&gt;AtmosphereServlet&lt;/servlet-name&gt;
        &lt;servlet-class&gt;org.atmosphere.cpr.AtmosphereServlet&lt;/servlet-class&gt;
        &lt;load-on-startup&gt;0 &lt;/load-on-startup&gt;
    &lt;/servlet&gt;
    &lt;servlet-mapping&gt;
         &lt;servlet-name&gt;AtmosphereServlet&lt;/servlet-name&gt;
         &lt;url-pattern&gt;/Atmosphere &lt;/url-pattern&gt;
    &lt;/servlet-mapping>
 * </code></pre></p>
 * 
 * @author Jeanfrancois Arcand
 */
public class AtmosphereServlet extends AbstractAsyncServlet 
        implements CometProcessor, HttpEventServlet{

    // If we detect Servlet 3.0, should we still use the default 
    // native Comet API.
    private boolean useNativeImplementation = true;
    
    public final static String SERVLET_30 = "javax.servlet.AsyncListener";
    public final static String GLASSFISH_V2 = 
            "com.sun.enterprise.web.connector.grizzly.comet.CometEngine";
    public final static String TOMCAT = 
            "org.apache.coyote.http11.Http11NioProcessor";
    public final static String JBOSS_5 = 
            "org.apache.coyote.http11.Http11AprProtocol";    
    public final static String JETTY = "org.mortbay.util.ajax.Continuation";  
    public final static String JETTY_7 = "org.eclipse.jetty.continuation.Continuation";
    public final static String GRIZZLY = "com.sun.grizzly.http.servlet.ServletAdapter";
    public final static String GLASSFISH_V3 = "com.sun.grizzly.comet.CometEngine";   
    public final static String WEBLOGIC = "weblogic.servlet.http.FutureResponseModel";
    public final static String JBOSSWEB="org.jboss.servlet.http.HttpEventFilter";

    public final static Logger logger = LoggerUtils.getLogger();

    private ArrayList<String> possibleAtmosphereHandlersCandidate = new ArrayList<String>();

   /**
     * The list of {@link AtmosphereHandler} and their associated mapping.
     */
    protected Map<String,AtmosphereHandlerWrapper> atmosphereHandlers = new HashMap<String, AtmosphereHandlerWrapper>();

    // The WebServer we are running on.
    protected CometSupport cometSupport;

    protected AtmosphereConfig config;

    // Simple struc for storing Grizzly and its associated Broadcaster.
    public static class AtmosphereHandlerWrapper{

        public AtmosphereHandler atmosphereHandler;
        public Broadcaster broadcaster;

        public AtmosphereHandlerWrapper(AtmosphereHandler g){
            this.atmosphereHandler = g;
            this.broadcaster = new DefaultBroadcaster();
        }
    }


    public class AtmosphereConfig{

        protected AtmosphereHandler ah;

        private BroadcasterLookup bl = new BroadcasterLookup();

        protected Map<String,AtmosphereHandlerWrapper> handlers(){
            return AtmosphereServlet.this.atmosphereHandlers;
        }

        public ServletContext getServletContext(){
            return AtmosphereServlet.this.getServletContext();
        }


        public String getInitParameter(String name){
            return AtmosphereServlet.this.getInitParameter(name);
        }


        public Enumeration getInitParameterNames(){
            return AtmosphereServlet.this.getInitParameterNames();
        }


        public ServletConfig getServletConfig(){
            return AtmosphereServlet.this.getServletConfig();
        }

        public String getWebServerName(){
            return AtmosphereServlet.this.cometSupport.getContainerName();
        }

        public String getWebServerVersion(){
            return AtmosphereServlet.this.cometSupport.getContainerDottedVersion();
        }

        /**
         * Return the {@link AtmosphereHandler} associated with this config.
         * @return the {@link AtmosphereHandler} associated with this config.
         */
        public AtmosphereHandler getAtmosphereHandler(){
            return ah;
        }

        /**
         * Return an instance of a {@link BroadcasterLookup}
         * @return an instance of a {@link BroadcasterLookup}
         */
        public BroadcasterLookup getBroadcasterLookup(){
            return bl;
        }
    }



    /**
     * Simple class/struck that hold the current state.
     */
    public static class Action{
        
        public enum TYPE {SUSPEND, RESUME, NONE, CANCEL}
        
        public long timeout = -1L;
        
        public TYPE type;
        
        public Action(){
            type = TYPE.NONE;
        }
        
        public Action(TYPE type){
            this.type = type;
        }
        
        public Action(TYPE type,long timeout){
            this.timeout = timeout;
            this.type = type;
        }
    }
    
    /**
     * Create an Atmosphere Servlet.
     */
    public AtmosphereServlet(){
        if (System.getProperty("org.atmosphere.useNative") != null){
            useNativeImplementation = Boolean
                    .parseBoolean(System.getProperty("org.atmosphere.useNative"));
        }
    }
    
    /**
     * Add an {@link AtmosphereHandler} serviced by the {@link Servlet}
     * This API is exposed to allow embedding an Atmosphere application.
     * @param The servlet mapping (servlet path)
     * @prama an implementation of an {@link AtmosphereHandler}
     */
    public void addAtmosphereHandler(String mapping, AtmosphereHandler h){
        if (mapping.startsWith("/")){
            mapping = "/" + mapping;
        }
        atmosphereHandlers.put(mapping,new AtmosphereHandlerWrapper(h));
    }
    
    /**
     * Load the {@link AtmosphereHandler} associated with this AtmosphereServlet.
     * @param sc the {@link ServletContext}
     */
    @Override
    public void init(ServletConfig sc) throws ServletException{
        super.init(sc);
        try{
            //TODO -> Add support for WEB-INF/lib/*.jar
            URL url = sc.getServletContext().getResource("/WEB-INF/classes/");
            URLClassLoader urlC = new URLClassLoader(new URL[] {url}, 
                    Thread.currentThread().getContextClassLoader());
            loadAtmosphereDotXml(sc.getServletContext().
                    getResourceAsStream("/META-INF/atmosphere.xml"), urlC);
            if (atmosphereHandlers.size() == 0){
                logger.warning("Missing META-INF/atmosphere.xml." +
                    " Will try to autodetect AtmosphereHandler");
                autoDetectAtmosphereHandlers(sc.getServletContext(), urlC);
            }
        } catch (Throwable t){
            throw new ServletException(t);
        }
        autoDetectContainer();
        
        cometSupport.init(sc);
        initAtmosphereServletProcessor(sc);
    }

    /**
     * Initialize {@link AtmosphereServletProcessor}
     * @param sc the {@link ServletConfig}
     * @throws javax.servlet.ServletException
     */
    void initAtmosphereServletProcessor(ServletConfig sc) throws ServletException{
        AtmosphereHandler a;
        for (Entry<String,AtmosphereHandlerWrapper> h: atmosphereHandlers.entrySet()){
            a = h.getValue().atmosphereHandler;
            if (a instanceof AtmosphereServletProcessor){
                ((AtmosphereServletProcessor)a).init(sc);
            }
        }
    }


    @Override
    public void destroy(){
        AtmosphereHandler a;
        for (Entry<String,AtmosphereHandlerWrapper> h: atmosphereHandlers.entrySet()){
            a = h.getValue().atmosphereHandler;
            h.getValue().broadcaster.destroy();
            config.getBroadcasterLookup().remove(h.getValue().broadcaster);
            if (a instanceof AtmosphereServletProcessor){
                ((AtmosphereServletProcessor)a).destroy();
            }
        }
    }

    /**
     * Load AtmosphereHandler defined under META-INF/atmosphere.xml
     */
    protected void loadAtmosphereDotXml(InputStream stream, URLClassLoader c) 
            throws IOException{

        if (stream == null){
            return;
        }

        AtmosphereConfigReader reader = new AtmosphereConfigReader(stream);

        Map<String, String> atmosphereHandlerNames = reader.getAtmosphereHandlers();
        Set<Entry<String,String>> entries = atmosphereHandlerNames.entrySet();
        for (Entry<String,String> entry: entries){
            AtmosphereHandler g;
            try {
                if (!entry.getValue().equals(ReflectorServletProcessor.class.getName())){
                    g = (AtmosphereHandler) c.loadClass(entry.getValue()).newInstance();
                } else {
                    g = new ReflectorServletProcessor();
                }
                logger.info("Sucessfully loaded " + g
                    + " mapped to context-path " + entry.getKey());

                AtmosphereHandlerWrapper wrapper = new AtmosphereHandlerWrapper(g);
                atmosphereHandlers.put(entry.getKey(), wrapper);
                for (Property p: reader.getProperty(entry.getValue())){
                    IntrospectionUtils.setProperty(g,p.name,p.value);
                }

                String broadcasterClass = reader.getBroadcasterClass(entry.getKey());
                if (broadcasterClass != null){
                    wrapper.broadcaster = (Broadcaster)
                            c.loadClass(broadcasterClass).newInstance();
                }

            } catch (Throwable t) {
                logger.log(Level.WARNING, "Unable to load AtmosphereHandler class: "
                        + entry.getValue(), t);
            }
        }
    }
    
    /**
     * Auto detect the underlying Servlet Container we are running on.
     */
    protected void autoDetectContainer(){
        config = new AtmosphereConfig();
        try{
            if (useNativeImplementation){
                logger.info("Forcing the use of the container's" +
                        " native Comet API implementation instead of Servlet 3.0");
                throw new ClassNotFoundException("Skip");
            }         
            Class.forName(SERVLET_30);
            cometSupport = new Servlet30Support(config);
        } catch (ClassNotFoundException e){
            try {
                Class.forName(GLASSFISH_V3);
                cometSupport = new GlassFishv3CometSupport(config);
            } catch (ClassNotFoundException ex) {
                try {
                    // GlassFish v2
                    Class.forName(JETTY);
                    cometSupport = new JettyCometSupport(config);
                } catch (ClassNotFoundException ex2) {
                    // Try JBOSSWeb first
                    try{
                        Class.forName(JBOSSWEB);
                        cometSupport = new JBossWebCometSupport(config);
                    } catch (ClassNotFoundException ex10) {
                        try{
                            Class.forName(TOMCAT);
                            cometSupport = new TomcatCometSupport(config);
                        } catch (ClassNotFoundException ex3) {
                            try{
                                Class.forName(GRIZZLY);
                                cometSupport = new GrizzlyCometSupport(config);
                            } catch (ClassNotFoundException ex4) {
                                try{
                                    Class.forName(GLASSFISH_V2);
                                    cometSupport = new GlassFishv2CometSupport(config);
                                } catch (ClassNotFoundException ex5){
                                    try{
                                        Class.forName(WEBLOGIC);
                                        cometSupport = new WebLogicCometSupport(config);
                                    } catch (ClassNotFoundException ex6){
                                        try{
                                            Class.forName(JETTY_7);
                                            cometSupport = new Jetty7CometSupport(config);
                                        } catch (ClassNotFoundException ex7){
                                            try{
                                                Class.forName(JBOSS_5);
                                                cometSupport = new TomcatCometSupport(config);
                                            } catch (ClassNotFoundException ex8){
                                                cometSupport = new BlockingIOCometSupport(config);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        
        logger.info("Atmosphere Framework running under container "
                + cometSupport.getContainerName()
                + " version " + cometSupport.getContainerDottedVersion());
    }

    /**
     * Auto detect instance of {@link AtmosphereHandler} in case META-INF/atmosphere.xml
     * is missing.
     * @param sc {@link ServletContext}
     * @param c {@link URLClassLoaser} to load the class.
     */
    protected void autoDetectAtmosphereHandlers(ServletContext sc, URLClassLoader c)
            throws MalformedURLException, URISyntaxException{
        String webInf = "/WEB-INF/classes/";
        String s = sc.getRealPath(webInf);

        // Weblogic bug
        if (s == null){
            URL u = sc.getResource(webInf);
            s = u.getPath();
        }
        
        File f = new File(s);
        if (f.isDirectory()){
            getFiles(f);
            for (String className: possibleAtmosphereHandlersCandidate){
                try {
                    className = className.substring(className.indexOf(webInf)
                            + webInf.length(), className.lastIndexOf(".")).replace('/','.');
                    AtmosphereHandler g = (AtmosphereHandler) c.loadClass(className).newInstance();
                    logger.info("Sucessfully loaded " + g
                        + " mapped to context-path " + g.getClass().getSimpleName());
                    atmosphereHandlers.put("/" + g.getClass().getSimpleName(), new AtmosphereHandlerWrapper(g));
                } catch (Throwable t) {
                    logger.finest(className + " is not a AtmosphereHandler");
                }
            }
        }
    }

    /**
     * Get the list of possible candidate to load as {@link AtmosphereHandler}
     * @param f the real path {@link File}
     */
    void getFiles(File f){
        File[] files = f.listFiles();
        for (File test: files){
            if (test.isDirectory()){
                getFiles(test);
            } else {
                String clazz = test.getAbsolutePath();
                if (clazz.endsWith(".class")){
                    possibleAtmosphereHandlersCandidate.add(clazz);
                }
            }
        }
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doHead(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doOptions(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doTrace(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
  
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doDelete(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doPut(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }   
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doGet(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        doPost(req, res);
    }
    
    /**
     * Delegate the request processing to an instance of {@link CometSupport}
     * @param req the {@link HttpServletRequest}
     * @param res  the {@link HttpServletResponse}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    @Override
    public void doPost(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        try{
            cometSupport.service(req, res);
        } catch (IllegalStateException ex){
            if (ex.getMessage() != null && ex.getMessage().startsWith("Tomcat failed")){
                logger.warning(ex.getMessage());
                logger.warning("Using the BlockingIOCometSupport.");
                cometSupport = new BlockingIOCometSupport(config);
                service(req,res);
            } else {
                logger.log(Level.SEVERE,"AtmosphereServlet exception", ex);
                throw ex;
            }
        }
    }

    /**
     * Hack to support Tomcat AIO like other WebServer. This method is invoked
     * by Tomcat when it detect a {@link Servlet} implements the interface
     * {@link CometProcessor} without invoking {@link Servlet#service}
     * 
     * @param cometEvent the {@link CometEvent}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    public void event(CometEvent cometEvent) throws IOException, ServletException {
        HttpServletRequest req = cometEvent.getHttpServletRequest();
        HttpServletResponse res = cometEvent.getHttpServletResponse();
        req.setAttribute(TomcatCometSupport.COMET_EVENT, cometEvent);
        cometSupport.service(req, res);
    }

    /**
     * Hack to support JBossWeb AIO like other WebServer. This method is invoked
     * by Tomcat when it detect a {@link Servlet} implements the interface
     * {@link HttpEventServlet} without invoking {@link Servlet#service}
     *
     * @param httpEvent the {@link CometEvent}
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    public void event(HttpEvent httpEvent) throws IOException, ServletException {
        HttpServletRequest req = httpEvent.getHttpServletRequest();
        HttpServletResponse res = httpEvent.getHttpServletResponse();
        req.setAttribute(JBossWebCometSupport.HTTP_EVENT, httpEvent);
        cometSupport.service(req, res);
    }

    /**
     * Weblogic specific comet based implementation.
     * 
     * @param rrk
     * @return
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    protected boolean doRequest(RequestResponseKey rrk) throws IOException, ServletException {
        try{
            rrk.getRequest().getSession().setAttribute(WebLogicCometSupport.RRK,rrk);
            Action action = cometSupport.service(rrk.getRequest(), rrk.getResponse());
            if (action.type == Action.TYPE.SUSPEND){
                if (action.timeout == -1){
                        rrk.setTimeout(Integer.MAX_VALUE);
                } else {
                    rrk.setTimeout((int)action.timeout);
                }
            }
            return action.type == Action.TYPE.SUSPEND;
        } catch (IllegalStateException ex){
            logger.log(Level.SEVERE,"AtmosphereServlet.doRequest exception", ex);
            throw ex;
        }        
    }
    
    /**
     * Weblogic specific comet based implementation.
     * 
     * @param rrk
     * @return
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    protected void doResponse(RequestResponseKey rrk, Object context) 
            throws IOException, ServletException {
        rrk.getResponse().flushBuffer();
    }
   
    /**
     * Weblogic specific comet based implementation.
     * 
     * @param rrk
     * @return
     * @throws java.io.IOException
     * @throws javax.servlet.ServletException
     */
    protected void doTimeout(RequestResponseKey rrk) throws IOException, ServletException {
        ((AsynchronousProcessor)cometSupport).timedout(rrk.getRequest(), rrk.getResponse());
    }


}
