/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.cpr;

import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.ExecutorService;

/**
 * A Broadcaster is responsible for delivering messages to its subscribed
 * {@link AtmosphereEvent}, which are representing a suspended response.
 * {@link AtmosphereEvent} can be added using {@link Broadcaster#addAtmosphereEvent},
 * so when {@link #broadcast(java.lang.Object)} execute,
 * {@link AtmosphereHandler#onMessage(org.atmosphere.cpr.AtmosphereEvent)} will
 * by invoked and the suspended connection will have a chance to write the
 * message available using {@link AtmosphereEvent#getMessage()}
 * <br>
 * A {@link Broadcaster}, by default, will use an {@link ExecutorService}, and the
 * number of Thread will be computed based on the core/cpu of the OS under
 * which the application run.
 * 
 * @author Jeanfrancois Arcand
 */
public interface Broadcaster<F> {

    public enum SCOPE { REQUEST, APPLICATION, VM }

    /**
     * Broadcast the {@link F} to all suspended response, e.g. invoke
     * {@link AtmosphereHandler#onMessage}.
     * 
     * @param o and {@link F} to be shared.
     * @return the final {@link F}, which might have been transformed by {@link MessageTransformer}
     */
    public F broadcast(F o);

    /**
     * Broadcast the {@link F} to all suspended response, e.g. invoke
     * {@link AtmosphereHandler#onMessage} with an instance of {@link AtmosphereEvent}, representing
     * a set of {@link AtmosphereHandler}.
     *
     * @param o and {@link F} to be shared.
     * @param event an {@link AtmosphereEvent}
     * @return the final {@link F}, which might have been transformed by {@link MessageTransformer}
     */
    public F broadcast(F o, AtmosphereEvent event);

    /**
     * Broadcast the {@link F} to all suspended response, e.g. invoke
     * {@link AtmosphereHandler#onMessage} with a {@link Set} of  {@link AtmosphereEvent},
     * representing a set of {@link AtmosphereHandler}.
     *
     * @param o and {@link F} to be shared.
     * @param event an {@link AtmosphereEvent}
     * @return the final {@link F}, which might have been transformed by {@link MessageTransformer}
     */
    public F broadcast(F o, Set<AtmosphereEvent> subset);

    /**
     * Add a {@link AtmosphereEvent} to the list of item to be notified when
     * the {@link Broadcaster#broadcast} is invoked.
     *
     * @param event an {@link AtmosphereEvent}
     * @return {@link AtmosphereEvent} if added, or null if it was already there.
     */
    public AtmosphereEvent addAtmosphereEvent(AtmosphereEvent event);

    /**
     * Remove a {@link AtmosphereEvent} from the list of item to be notified when
     * the {@link Broadcaster#broadcast} is invoked.
     *
     * @param event an {@link AtmosphereEvent}
     * @return {@link AtmosphereEvent} if removed, or null if it was not.
     */
    public AtmosphereEvent removeAtmosphereEvent(AtmosphereEvent event);

    /**
     * Set the {@link BroadcasterConfig} instance.
     * @param bf
     */
    public void setBroadcasterConfig(BroadcasterConfig bc);

    /**
     * Return the current {@link BroadcasterConfig}
     * @return the current {@link BroadcasterConfig}
     */
    public BroadcasterConfig getBroadcasterConfig();

    /**
     * Destroy this instance and shutdown it's associated {@link ExecutorServices}
     */
    public void destroy();

    /**
     * Return an Iterator<AtmosphereEvent> of {@link AtmosphereEvent}
     */
    public Iterator<AtmosphereEvent> getAtmosphereEvents();

    /**
     * Set the {@link Broadcaster#SCOPE}
     */
    public void setScope(SCOPE scope);

    /**
     * Retunr the {@link Broadcaster#SCOPE}
     * @return  {@link Broadcaster#SCOPE}
     */
    public SCOPE getScope();

    /**
     * Set the name of this {@link Broadcaster}
     * @param name of this {@link Broadcaster}
     */
    public void setName(String name);

    /**
     * Return the name of this {@link Broadcaster}
     * @return the name of this {@link Broadcaster}
     */
    public String getName();

}
