/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.atmosphere.cpr;

import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.atmosphere.util.LoggerUtils;

/**
 * Handle {@link Broadcaster} configuration like {@link ExecutorService} and
 * {@link MessageTransformer}
 * 
 * @author Jeanfrancois Arcand
 */
public class BroadcasterConfig<F> {

    private final static int numOfProcessor;
    static{
        numOfProcessor = Runtime.getRuntime().availableProcessors();
    }

    private ConcurrentLinkedQueue<MessageTransformer<F>> transformers =
            new ConcurrentLinkedQueue<MessageTransformer<F>>();

    private ExecutorService executorService = Executors.newFixedThreadPool(numOfProcessor);

    private ExecutorService defaultExecutorService = executorService;

    public BroadcasterConfig(){
        LoggerUtils.getLogger().info("DefaultBroadcaster " +
                "configured using a Thread Pool of size: " + numOfProcessor);
    }


    /**
     * Set an {@link ExecutorService} which can be used to dispatch
     * {@link AtmosphereEvent}. By default, an {@Executors#newSingleThreadExecutor}
     * is used if that method is not invoked.
     * @param executorService to be used when broadcasting.
     */
    public void setExecutorService(ExecutorService executorService) {
        this.executorService = executorService;
    }

    /**
     * Return the {@link ExecutorService} this {@link Broadcaster} support.
     * By defaul it returns {@Executors#newSingleThreadExecutor}, which means
     * a single Thread is used to broadcast.
     * @return An ExecutorService.
     */
    public ExecutorService getExecutorService() {
        return executorService;
    }

    /**
     * Add a {@link MessageTransformer}
     * @param e {@link MessageTransformer}
     * @return true if added.
     */
    public boolean addMessageTransformer(MessageTransformer<F> e){
        if (transformers.contains(e)) return false;
        return transformers.offer(e);
    }

    /**
     * Remove a {@link MessageTransformer}
     * @param e {@link MessageTransformer}
     * @return true if removed
     */
    public boolean removeMessageTransformer(MessageTransformer<F> e){
        return transformers.remove(e);
    }

    /**
     * Invoke {@link MessageTransformer} in the other they were added.
     */
    Object transform(F object){
        F transformed = object;
        for (MessageTransformer<F> mf: transformers){
            transformed = mf.transform(transformed);
        }
        return transformed;
    }

    /**
     * Return the default {@link ExecutorService}.
     * @return the defaultExecutorService
     */
    public ExecutorService getDefaultExecutorService() {
        return defaultExecutorService;
    }
}
