/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.cpr;

import java.io.IOException;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import org.atmosphere.cpr.Broadcaster.SCOPE;
import org.atmosphere.util.BroadcasterLookup;
import org.atmosphere.util.LoggerUtils;

/**
 * {@link Broadcaster} implementation.
 *
 * Broadcast messages to suspended reponse using the caller's Thread.
 * This basic {@link Broadcaster} isn't using any {@link ExecutorService} to
 * broadcast messages, so it is single threaded. 
 * 
 * @author Jeanfrancois Arcand
 */
public class DefaultBroadcaster implements Broadcaster<String> {

    private ConcurrentLinkedQueue<AtmosphereEvent> events =
            new ConcurrentLinkedQueue<AtmosphereEvent>();

    private BroadcasterConfig<String> bc;

    private final BlockingQueue<Entry> messages =
            new LinkedBlockingQueue<Entry>();

    private final AtomicBoolean started = new AtomicBoolean(false);

    private SCOPE scope = SCOPE.APPLICATION;

    private String name = DefaultBroadcaster.class.getSimpleName();
   
    public DefaultBroadcaster() {
        bc = new BroadcasterConfig<String>();
    }

    /**
     * {@inheritDoc}
     */
    public void destroy() {
        getBroadcasterConfig().getExecutorService().shutdown();
        getBroadcasterConfig().getDefaultExecutorService().shutdown();
    }

    /**
     * {@inheritDoc}
     */
    public Iterator<AtmosphereEvent> getAtmosphereEvents() {
        return events.iterator();
    }

    /**
     * {@inheritDoc}
     */
    public void setScope(SCOPE scope) {
        this.scope = scope;
    }

    /**
     * {@inheritDoc}
     */
    public SCOPE getScope() {
        return scope;
    }

    /**
     * {@inheritDoc}
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * {@inheritDoc}
     */
    public String getName() {
        return name;
    }

    private class Entry{
        String message;
        Object eventsToPush;

        Entry(String message,Object eventsToPush){
            this.message = message;
            this.eventsToPush = eventsToPush;
        }
    }

    void start(){
        if (!started.getAndSet(true)) {
            bc.getExecutorService().submit(new Runnable() {
                public void run() {
                    try {
                        Entry msg = messages.take();
                        // Leader/follower
                        bc.getExecutorService().submit(this);
                        if (msg.eventsToPush == null){
                            for (AtmosphereEvent event : events) {
                                synchronized(event){
                                    if (event instanceof AtmosphereEventImpl){
                                        ((AtmosphereEventImpl)event).setMessage(msg.message);
                                    }
                                    broadcast(event);
                                }
                            }
                        } else if (msg.eventsToPush instanceof AtmosphereEvent){
                            AtmosphereEvent event = (AtmosphereEvent)msg.eventsToPush;
                            synchronized(event){
                                if (event instanceof AtmosphereEventImpl){
                                    ((AtmosphereEventImpl)event).setMessage(msg.message);
                                }
                                broadcast(event);
                            }
                        } else if (msg.eventsToPush instanceof Set){
                            Set<AtmosphereEvent> events =
                                    (Set<AtmosphereEvent>)msg.eventsToPush;
                            for (AtmosphereEvent event : events) {
                                synchronized(event){
                                    if (event instanceof AtmosphereEventImpl){
                                        ((AtmosphereEventImpl)event).setMessage(msg.message);
                                    }
                                    broadcast(event);
                                }
                            }
                        }
                    } catch (InterruptedException ex) {
                        LoggerUtils.getLogger().log(Level.SEVERE, null, ex);
                    }
                }
            });
        }
    }

    private void broadcast(final AtmosphereEvent event){
        try {
            event.getAtmosphereConfig().getAtmosphereHandler().onMessage(event);
        } catch (IOException ex){
            // AtmosphereHandler.onException?
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
        } catch (IllegalStateException ex){
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
            removeAtmosphereEvent(event);
        }
    }

    /**
     * {@inheritDoc}
     */
    public String broadcast(String msg) {
        start();
        msg= (String)bc.transform(msg);
        messages.offer(new Entry(msg,null));
        return msg;
    }

    /**
     * {@inheritDoc}
     */
    public String broadcast(String msg, AtmosphereEvent event) {
        start();
        msg= (String)bc.transform(msg);
        messages.offer(new Entry(msg,event));
        return msg;
    }
    
    /**
     * {@inheritDoc}
     */
    public String broadcast(String msg, Set<AtmosphereEvent> subset) {
        start();
        msg= (String)bc.transform(msg);
        messages.offer(new Entry(msg,subset));
        return msg;
    }

    /**
     * {@inheritDoc}
     */
    public AtmosphereEvent addAtmosphereEvent(AtmosphereEvent event) {
        if (events.contains(event)) return null;
        events.offer(event);
        event.getAtmosphereConfig().getBroadcasterLookup().add(this);
        return event;
    }
    
    /**
     * {@inheritDoc}
     */
    public AtmosphereEvent removeAtmosphereEvent(AtmosphereEvent event) {
        if (!events.contains(event)) return null;
        events.remove(event);
        return event;
    }

    /**
     * Set the {@link BroadcasterConfig} instance.
     * @param bf
     */
    public void setBroadcasterConfig(BroadcasterConfig bc){
        this.bc = bc;
    }

    /**
     * Return the current {@link BroadcasterConfig}
     * @return the current {@link BroadcasterConfig}
     */
    public BroadcasterConfig getBroadcasterConfig(){
        return bc;
    }

}
