/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.atmosphere.handler;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.atmosphere.cpr.AtmosphereEvent;
import org.atmosphere.cpr.AtmosphereHandler;
import org.atmosphere.cpr.AtmosphereServletProcessor;

/**
 * Simple {@link AtmosphereServletProcessor} that delegate the {@link AtmosphereHandler#onEvent}
 * to the {@link Servlet#service(javax.servlet.ServletRequest, javax.servlet.ServletResponse)
 * and storing the {@link AtmosphereEvent} as a {@link HttpServletRequest} attribute named
 * org.atmosphere.cpr.AtmosphereEvent. The {@link AtmosphereEvent} can later be retrieved
 * and used to supend/resume and broadcast
 *
 * @author Jeanfrancois Arcand
 */
public class ReflectorServletProcessor extends AbstractReflectorAtmosphereHandler
        implements AtmosphereServletProcessor {

    private String servletClass;
    private Servlet servlet;
    private ServletWrapper wrapper;

    public final static String ATMOSPHERE_EVENT =
            org.atmosphere.cpr.AtmosphereEvent.class.getName();
    public final static String ATMOSPHERE_HANDLER =
            org.atmosphere.handler.ReflectorServletProcessor.class.getName();

    public ReflectorServletProcessor(){}

    void loadServlet(ServletContext sc) throws MalformedURLException,
            InstantiationException, IllegalAccessException, ClassNotFoundException {
        URL url = sc.getResource("/WEB-INF/lib/");
        URLClassLoader urlC = new URLClassLoader(new URL[]{url},
                Thread.currentThread().getContextClassLoader());
        try{
            servlet = (Servlet) urlC.loadClass(getServletClass()).newInstance();
        } catch (NullPointerException ex){
            // We failed to load the servlet, let's try directly.
            servlet = (Servlet)Thread.currentThread().getContextClassLoader()
                    .loadClass(servletClass).newInstance();
        }
    }

    /**
     * Delegate the request to the Servlet.service method, and add the {@link AtmosphereEvent}
     * to the {@link HttpServletRequest#setAttribute(java.lang.String, java.lang.Object))}.
     * The {@link AtmosphereEvent} can ve retrieved using {@link #ATMOSPHERE_EVENT}
     * value.
     * @param event The {@link AtmosphereEvent}
     * @return The {@link AtmosphereEvent}
     * @throws java.io.IOException
     */
    public AtmosphereEvent onEvent(AtmosphereEvent<HttpServletRequest, HttpServletResponse> event)
            throws IOException {
        event.getRequest().setAttribute(ATMOSPHERE_EVENT, event);
        event.getRequest().setAttribute(ATMOSPHERE_HANDLER, this);
        try {
            wrapper.service(event.getRequest(), event.getResponse());
        } catch (ServletException ex) {
            throw new RuntimeException(ex);
        }
        return event;
    }

    public void init(ServletConfig sc) throws ServletException {
        try {
            loadServlet(sc.getServletContext());
        } catch (Exception ex) {
            throw new ServletException(ex);
        }
        servlet.init(sc);
        wrapper = new ServletWrapper();
    }

    public void destroy() {
        if (servlet != null) {
            servlet.destroy();
        }
    }

    /**
     * Set the Servlet class.
     * @return the servletClass
     */
    public String getServletClass() {
        return servletClass;
    }

    /**
     * Return the Servlet class name.
     * @param servletClass the servletClass to set
     */
    public void setServletClass(String servletClass) {
        this.servletClass = servletClass;
    }

    /**
     * Simple wrapper around a {@link Servlet}
     */
    private class ServletWrapper extends HttpServlet {

        @Override
        public void destroy() {
            servlet.destroy();
        }
        @Override
        public String getInitParameter(String name) {
            return getServletConfig().getInitParameter(name);
        }
        @Override
        public Enumeration getInitParameterNames() {
            return getServletConfig().getInitParameterNames();
        }
        @Override
        public ServletConfig getServletConfig() {
            return servlet.getServletConfig();
        }
        @Override
        public ServletContext getServletContext() {
            return getServletConfig().getServletContext();
        }
        @Override
        public String getServletInfo() {
            return servlet.getServletInfo();
        }

        @Override
        public void init(ServletConfig config) throws ServletException {
        }
        @Override
        public void init() throws ServletException {
        }

        @Override
        public void log(String msg) {
            getServletContext().log(getServletName() + ": " + msg);
        }

        @Override
        public void log(String message, Throwable t) {
            getServletContext().log(getServletName() + ": " + message, t);
        }


        @Override
        public void service(ServletRequest req, ServletResponse res)
                throws ServletException, IOException{
            servlet.service(req, res);
        }

        @Override
        public String getServletName() {
            return servlet.getServletConfig().getServletName();
        }

    }
}
