/*
 * Copyright 2008-2021 Async-IO.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package org.atmosphere.interceptor;

import org.atmosphere.cpr.Action;
import org.atmosphere.cpr.AtmosphereInterceptorAdapter;
import org.atmosphere.cpr.AtmosphereResource;
import org.atmosphere.util.Utils;

import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * An interceptor that create an {@link jakarta.servlet.http.HttpSession} on the fist received request so transport like long-polling
 * can share the session with request coming after the suspend operation. Without this and because
 * with long-polling cookies aren't read by the browser until the response is resumed, the session id will not
 * be the same so session couldn't be used.
 *
 * @author Jeanfrancois Arcand
 */
public class SessionCreationInterceptor extends AtmosphereInterceptorAdapter {

    // This can cause memory leak.
    private final ConcurrentLinkedQueue<String> ids = new ConcurrentLinkedQueue<>();

    @Override
    public Action inspect(AtmosphereResource r) {

        if (Utils.webSocketMessage(r)) return Action.CONTINUE;

        if (r.session(false) == null
                && !ids.remove(r.uuid())
                && r.getRequest().getMethod().equalsIgnoreCase("GET")) {
            r.session(true);
            ids.offer(r.uuid());
            return Action.CANCELLED;
        }
        return Action.CONTINUE;
    }
}

