/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.ext.web.handler.impl;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.MultiMap;
import io.vertx.core.Vertx;
import io.vertx.core.VertxException;
import io.vertx.core.file.FileProps;
import io.vertx.core.file.FileSystem;
import io.vertx.core.file.FileSystemException;
import io.vertx.core.http.HttpMethod;
import io.vertx.core.http.HttpServerRequest;
import io.vertx.core.json.JsonArray;
import io.vertx.core.logging.Logger;
import io.vertx.core.logging.LoggerFactory;
import io.vertx.ext.web.RoutingContext;
import io.vertx.ext.web.handler.StaticHandler;
import io.vertx.ext.web.impl.LRUCache;
import io.vertx.ext.web.impl.Utils;
import java.io.File;
import java.nio.file.NoSuchFileException;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class StaticHandlerImpl
implements StaticHandler {
    private static final Logger log = LoggerFactory.getLogger(StaticHandlerImpl.class);
    private final DateFormat dateTimeFormatter = Utils.createISODateTimeFormatter();
    private Map<String, CacheEntry> propsCache;
    private String webRoot = "webroot";
    private long maxAgeSeconds = 86400L;
    private boolean directoryListing = false;
    private String directoryTemplateResource = "vertx-web-directory.html";
    private String directoryTemplate;
    private boolean includeHidden = true;
    private boolean filesReadOnly = true;
    private boolean cachingEnabled = true;
    private long cacheEntryTimeout = 30000L;
    private String indexPage = "/index.html";
    private int maxCacheSize = 10000;
    private static int NUM_SERVES_TUNING_FS_ACCESS = 1000;
    private boolean alwaysAsyncFS = false;
    private long maxAvgServeTimeNanoSeconds = 1000000L;
    private boolean tuning = true;
    private long totalTime;
    private long numServesBlocking;
    private boolean useAsyncFS;
    private long nextAvgCheck = NUM_SERVES_TUNING_FS_ACCESS;

    public StaticHandlerImpl(String root) {
        this.setRoot(root);
    }

    public StaticHandlerImpl() {
    }

    private String directoryTemplate(Vertx vertx) {
        if (this.directoryTemplate == null) {
            this.directoryTemplate = Utils.readFileToString(vertx, this.directoryTemplateResource);
        }
        return this.directoryTemplate;
    }

    private void writeCacheHeaders(HttpServerRequest request, FileProps props) {
        MultiMap headers = request.response().headers();
        if (this.cachingEnabled) {
            headers.set("cache-control", "public, max-age=" + this.maxAgeSeconds);
            headers.set("last-modified", this.dateTimeFormatter.format(props.lastModifiedTime()));
        }
        headers.set("date", this.dateTimeFormatter.format(new Date()));
    }

    @Override
    public void handle(RoutingContext context) {
        HttpServerRequest request = context.request();
        if (request.method() != HttpMethod.GET && request.method() != HttpMethod.HEAD) {
            if (log.isTraceEnabled()) {
                log.trace("Not GET or HEAD so ignoring request");
            }
            context.next();
        } else {
            String path = context.normalisedPath();
            if (path == null) {
                log.warn("Invalid path: " + context.request().path() + " so returning 404");
                context.fail(404);
                return;
            }
            if (!this.directoryListing && "/".equals(path)) {
                path = this.indexPage;
            }
            this.sendStatic(context, path);
        }
    }

    private void sendStatic(RoutingContext context, String path) {
        int idx;
        String name;
        String file = null;
        if (!this.includeHidden && (name = (file = this.getFile(path, context)).substring((idx = file.lastIndexOf(47)) + 1)).length() > 0 && name.charAt(0) == '.') {
            context.fail(404);
            return;
        }
        CacheEntry entry = null;
        if (this.cachingEnabled && (entry = this.propsCache().get(path)) != null) {
            HttpServerRequest request = context.request();
            if ((this.filesReadOnly || !entry.isOutOfDate()) && entry.shouldUseCached(request)) {
                context.response().setStatusCode(304).end();
                return;
            }
        }
        if (file == null) {
            file = this.getFile(path, context);
        }
        if (this.filesReadOnly && entry != null) {
            FileProps props = entry.props;
            this.sendFile(context, file, props);
        } else {
            String sfile = file;
            this.getFileProps(context, file, res -> {
                if (res.succeeded()) {
                    FileProps fprops = (FileProps)res.result();
                    if (fprops == null) {
                        context.fail(404);
                    } else if (fprops.isDirectory()) {
                        this.sendDirectory(context, path, sfile);
                    } else {
                        this.propsCache().put(path, new CacheEntry(fprops, System.currentTimeMillis()));
                        this.sendFile(context, sfile, fprops);
                    }
                } else if (res.cause() instanceof NoSuchFileException) {
                    context.fail(404);
                } else {
                    context.fail(res.cause());
                }
            });
        }
    }

    private void sendDirectory(RoutingContext context, String path, String file) {
        if (this.directoryListing) {
            this.sendDirectoryListing(file, context);
        } else if (this.indexPage != null) {
            String indexPath = path.endsWith("/") && this.indexPage.startsWith("/") ? path + this.indexPage.substring(1) : (!path.endsWith("/") && !this.indexPage.startsWith("/") ? path + "/" + this.indexPage.substring(1) : path + this.indexPage);
            this.sendStatic(context, indexPath);
        } else {
            context.fail(403);
        }
    }

    private synchronized void getFileProps(RoutingContext context, String file, Handler<AsyncResult<FileProps>> resultHandler) {
        FileSystem fs = context.vertx().fileSystem();
        if (this.alwaysAsyncFS || this.useAsyncFS) {
            fs.props(file, resultHandler);
        } else {
            long start = 0L;
            if (this.tuning) {
                start = System.nanoTime();
            }
            try {
                FileProps props = fs.propsBlocking(file);
                if (this.tuning) {
                    long end = System.nanoTime();
                    long dur = end - start;
                    this.totalTime += dur;
                    ++this.numServesBlocking;
                    if (this.numServesBlocking == Long.MAX_VALUE) {
                        this.resetTuning();
                    } else if (this.numServesBlocking == this.nextAvgCheck) {
                        double avg = (double)this.totalTime / (double)this.numServesBlocking;
                        if (avg > (double)this.maxAvgServeTimeNanoSeconds) {
                            this.useAsyncFS = true;
                            log.info("Switching to async file system access in static file server as fs access is slow! (Average access time of " + avg + " ns)");
                            this.tuning = false;
                        }
                        this.nextAvgCheck += (long)NUM_SERVES_TUNING_FS_ACCESS;
                    }
                }
                resultHandler.handle(Future.succeededFuture(props));
            }
            catch (FileSystemException e) {
                resultHandler.handle(Future.failedFuture(e.getCause()));
            }
        }
    }

    private void resetTuning() {
        this.nextAvgCheck = NUM_SERVES_TUNING_FS_ACCESS;
        this.totalTime = 0L;
        this.numServesBlocking = 0L;
    }

    private void sendFile(RoutingContext context, String file, FileProps fileProps) {
        HttpServerRequest request = context.request();
        this.writeCacheHeaders(request, fileProps);
        if (request.method() == HttpMethod.HEAD) {
            request.response().end();
        } else {
            request.response().sendFile(file, res2 -> {
                if (res2.failed()) {
                    context.fail(res2.cause());
                }
            });
        }
    }

    @Override
    public StaticHandler setWebRoot(String webRoot) {
        this.setRoot(webRoot);
        return this;
    }

    @Override
    public StaticHandler setFilesReadOnly(boolean readOnly) {
        this.filesReadOnly = readOnly;
        return this;
    }

    @Override
    public StaticHandler setMaxAgeSeconds(long maxAgeSeconds) {
        if (maxAgeSeconds < 0L) {
            throw new IllegalArgumentException("timeout must be >= 0");
        }
        this.maxAgeSeconds = maxAgeSeconds;
        return this;
    }

    @Override
    public StaticHandler setMaxCacheSize(int maxCacheSize) {
        if (maxCacheSize < 1) {
            throw new IllegalArgumentException("maxCacheSize must be >= 1");
        }
        this.maxCacheSize = maxCacheSize;
        return this;
    }

    @Override
    public StaticHandler setCachingEnabled(boolean enabled) {
        this.cachingEnabled = enabled;
        return this;
    }

    @Override
    public StaticHandler setDirectoryListing(boolean directoryListing) {
        this.directoryListing = directoryListing;
        return this;
    }

    @Override
    public StaticHandler setDirectoryTemplate(String directoryTemplate) {
        this.directoryTemplateResource = directoryTemplate;
        this.directoryTemplate = null;
        return this;
    }

    @Override
    public StaticHandler setIncludeHidden(boolean includeHidden) {
        this.includeHidden = includeHidden;
        return this;
    }

    @Override
    public StaticHandler setCacheEntryTimeout(long timeout) {
        if (timeout < 1L) {
            throw new IllegalArgumentException("timeout must be >= 1");
        }
        this.cacheEntryTimeout = timeout;
        return this;
    }

    @Override
    public StaticHandler setIndexPage(String indexPage) {
        Objects.requireNonNull(indexPage);
        if (!indexPage.startsWith("/")) {
            indexPage = "/" + indexPage;
        }
        this.indexPage = indexPage;
        return this;
    }

    @Override
    public StaticHandler setAlwaysAsyncFS(boolean alwaysAsyncFS) {
        this.alwaysAsyncFS = alwaysAsyncFS;
        return this;
    }

    @Override
    public synchronized StaticHandler setEnableFSTuning(boolean enableFSTuning) {
        this.tuning = enableFSTuning;
        if (!this.tuning) {
            this.resetTuning();
        }
        return this;
    }

    @Override
    public StaticHandler setMaxAvgServeTimeNs(long maxAvgServeTimeNanoSeconds) {
        this.maxAvgServeTimeNanoSeconds = maxAvgServeTimeNanoSeconds;
        return this;
    }

    private Map<String, CacheEntry> propsCache() {
        if (this.propsCache == null) {
            this.propsCache = new LRUCache<String, CacheEntry>(this.maxCacheSize);
        }
        return this.propsCache;
    }

    private Date parseDate(String header) {
        try {
            return this.dateTimeFormatter.parse(header);
        }
        catch (ParseException e) {
            throw new VertxException(e);
        }
    }

    private String getFile(String path, RoutingContext context) {
        String file = this.webRoot + Utils.pathOffset(path, context);
        if (log.isTraceEnabled()) {
            log.trace("File to serve is " + file);
        }
        return file;
    }

    private void setRoot(String webRoot) {
        Objects.requireNonNull(webRoot);
        if (webRoot.startsWith("/")) {
            throw new IllegalArgumentException("root cannot start with '/'");
        }
        this.webRoot = webRoot;
    }

    private void sendDirectoryListing(String dir, RoutingContext context) {
        FileSystem fileSystem = context.vertx().fileSystem();
        HttpServerRequest request = context.request();
        fileSystem.readDir(dir, asyncResult -> {
            if (asyncResult.failed()) {
                context.fail(asyncResult.cause());
            } else {
                String accept = request.headers().get("accept");
                if (accept == null) {
                    accept = "text/plain";
                }
                if (accept.contains("html")) {
                    String normalizedDir = dir.substring(this.webRoot.length());
                    if (!normalizedDir.endsWith("/")) {
                        normalizedDir = normalizedDir + "/";
                    }
                    StringBuilder files = new StringBuilder("<ul id=\"files\">");
                    List list = (List)asyncResult.result();
                    Collections.sort(list);
                    for (String s : list) {
                        String file = s.substring(s.lastIndexOf(File.separatorChar) + 1);
                        if (!this.includeHidden && file.charAt(0) == '.') continue;
                        files.append("<li><a href=\"");
                        files.append(normalizedDir);
                        files.append(file);
                        files.append("\" title=\"");
                        files.append(file);
                        files.append("\">");
                        files.append(file);
                        files.append("</a></li>");
                    }
                    files.append("</ul>");
                    int slashPos = 0;
                    for (int i = normalizedDir.length() - 2; i > 0; --i) {
                        if (normalizedDir.charAt(i) != '/') continue;
                        slashPos = i;
                        break;
                    }
                    String parent = "<a href=\"" + normalizedDir.substring(0, slashPos + 1) + "\">..</a>";
                    request.response().putHeader("content-type", "text/html");
                    request.response().end(this.directoryTemplate(context.vertx()).replace("{directory}", normalizedDir).replace("{parent}", parent).replace("{files}", files.toString()));
                } else if (accept.contains("json")) {
                    JsonArray json = new JsonArray();
                    for (String s : (List)asyncResult.result()) {
                        String file = s.substring(s.lastIndexOf(File.separatorChar) + 1);
                        if (!this.includeHidden && file.charAt(0) == '.') continue;
                        json.add(file);
                    }
                    request.response().putHeader("content-type", "application/json");
                    request.response().end(json.encode());
                } else {
                    StringBuilder buffer = new StringBuilder();
                    for (String s : (List)asyncResult.result()) {
                        String file = s.substring(s.lastIndexOf(File.separatorChar) + 1);
                        if (!this.includeHidden && file.charAt(0) == '.') continue;
                        buffer.append(file);
                        buffer.append('\n');
                    }
                    request.response().putHeader("content-type", "text/plain");
                    request.response().end(buffer.toString());
                }
            }
        });
    }

    private final class CacheEntry {
        final FileProps props;
        long createDate;

        private CacheEntry(FileProps props, long createDate) {
            this.props = props;
            this.createDate = createDate;
        }

        boolean shouldUseCached(HttpServerRequest request) {
            String ifModifiedSince = request.headers().get("if-modified-since");
            if (ifModifiedSince == null) {
                return false;
            }
            Date ifModifiedSinceDate = StaticHandlerImpl.this.parseDate(ifModifiedSince);
            boolean modifiedSince = Utils.secondsFactor(this.props.lastModifiedTime()) > ifModifiedSinceDate.getTime();
            return !modifiedSince;
        }

        boolean isOutOfDate() {
            boolean outOfDate = System.currentTimeMillis() - this.createDate > StaticHandlerImpl.this.cacheEntryTimeout;
            return outOfDate;
        }
    }
}

