/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.samples.chat;

import java.io.IOException;
import java.util.logging.Logger;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.atmosphere.cpr.AtmosphereEvent;
import org.atmosphere.cpr.AtmosphereHandler;
import org.atmosphere.cpr.Broadcaster;

/**
 * Simple AtmosphereHandler that implement the logic to build a Chat application.
 *
 * @author Jeanfrancois Arcand
 * @autor TAKAI Naoto (Orginial author for the Comet based Chat).
 */
public class ChatAtmosphereHandler implements AtmosphereHandler<HttpServletRequest, HttpServletResponse> {

    private static final String BEGIN_SCRIPT_TAG = "<script type='text/javascript'>\n";
    private static final String END_SCRIPT_TAG = "</script>\n";
    private static final Logger logger = Logger.getLogger("Grizzly");
    private static final long serialVersionUID = -2919167206889576860L;

    public ChatAtmosphereHandler() {
    }

    /**
     * When the {@link AtmosphereServlet} detect an {@link HttpServletRequest}
     * maps to this {@link AtmosphereHandler}, the  {@link AtmosphereHandler#onEvent}
     * gets invoked and the response will be suspended depending on the http
     * method, e.g. GET will suspend the connection, POST will broadcast chat
     * message to suspended connection.
     *
     * @param event An {@link AtmosphereEvent}
     * @return {@link AtmosphereEvent}
     * @throws java.io.IOException
     */
    public AtmosphereEvent<HttpServletRequest, HttpServletResponse>
            onEvent(AtmosphereEvent<HttpServletRequest, HttpServletResponse> event) throws IOException {
        HttpServletRequest req = event.getRequest();
        HttpServletResponse res = event.getResponse();

        res.setContentType("text/html");
        res.addHeader("Cache-Control", "private");
        res.addHeader("Pragma", "no-cache");
        if (req.getMethod().equalsIgnoreCase("GET")) {
            // for IE
            res.getWriter().write("<!-- Comet is a programming technique that enables web " +
                    "servers to send data to the client without having any need " +
                    "for the client to request it. -->\n");
            res.getWriter().flush();

            //
            // Mark this connection as suspended.
            //
            event.suspend();
        } else if (req.getMethod().equalsIgnoreCase("POST")) {
            res.setCharacterEncoding("UTF-8");
            String action = req.getParameterValues("action")[0];
            String name = req.getParameterValues("name")[0];

            if ("login".equals(action)) {
                event.getBroadcaster().broadcast(
                        BEGIN_SCRIPT_TAG + toJsonp("System Message from "
                        + event.getWebServerName(), name + " has joined.") + END_SCRIPT_TAG);
                res.getWriter().write("success");
                res.getWriter().flush();
            } else if ("post".equals(action)) {
                String message = req.getParameterValues("message")[0];
                event.getBroadcaster().broadcast(BEGIN_SCRIPT_TAG + toJsonp(name, message) + END_SCRIPT_TAG);
                res.getWriter().write("success");
                res.getWriter().flush();
            } else {
                res.setStatus(422);

                res.getWriter().write("success");
                res.getWriter().flush();
            }
        }
        return event;
    }

    /**
     * Invoked when a call to {@link Broadcaster#broadcast(java.lang.Object)} is
     * issued or when the response times out, e.g whne the value {@link AtmosphereEvent#suspend(long)}
     * expires.
     *
     * @param event An {@link AtmosphereEvent}
     * @return {@link AtmosphereEvent}
     * @throws java.io.IOException
     */
    public AtmosphereEvent<HttpServletRequest, HttpServletResponse>
            onMessage(AtmosphereEvent<HttpServletRequest, HttpServletResponse> event) throws IOException {
        HttpServletRequest req = event.getRequest();
        HttpServletResponse res = event.getResponse();

        if (event.isResuming()) {
            String script = BEGIN_SCRIPT_TAG + "window.parent.app.listen();\n" + END_SCRIPT_TAG;

            res.getWriter().write(script);
            res.getWriter().flush();
        } else {
            res.getWriter().write(event.getMessage().toString());
            res.getWriter().flush();
        }
        return event;
    }

    private String escape(String orig) {
        StringBuffer buffer = new StringBuffer(orig.length());

        for (int i = 0; i < orig.length(); i++) {
            char c = orig.charAt(i);
            switch (c) {
                case '\b':
                    buffer.append("\\b");
                    break;
                case '\f':
                    buffer.append("\\f");
                    break;
                case '\n':
                    buffer.append("<br />");
                    break;
                case '\r':
                    // ignore
                    break;
                case '\t':
                    buffer.append("\\t");
                    break;
                case '\'':
                    buffer.append("\\'");
                    break;
                case '\"':
                    buffer.append("\\\"");
                    break;
                case '\\':
                    buffer.append("\\\\");
                    break;
                case '<':
                    buffer.append("&lt;");
                    break;
                case '>':
                    buffer.append("&gt;");
                    break;
                case '&':
                    buffer.append("&amp;");
                    break;
                default:
                    buffer.append(c);
            }
        }

        return buffer.toString();
    }

    private String toJsonp(String name, String message) {
        return "window.parent.app.update({ name: \"" + escape(name) + "\", message: \"" + escape(message) + "\" });\n";
    }
}
