/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.util;


import org.atmosphere.cpr.Broadcaster;

import java.util.Enumeration;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;

/**
 * A class that can be used to lookup {@link Broadcaster}
 * based on the returned value of {@link Broadcaster#getID()} and
 * {@link Broadcaster#getScope()}
 *
 * @author Jeanfrancois Arcand
 */
public class BroadcasterLookup{

    private final static ConcurrentHashMap<Object,Broadcaster> store
            = new ConcurrentHashMap<Object,Broadcaster>();

    /**
     * Lookup a {@link Broadcaster} instance using {@link Broadcaster#getID()} or ID
     * used when invoking {@link BroadcasterLookup#build(java.lang.Class, java.lang.Object));
     * @param id The Broadcaster's unique ID, or name.
     * @return a Broadcaster, or null if not found.
     */
    public final static Broadcaster lookup(Class<? extends Broadcaster> c, Object id){
        Broadcaster b = getBroadcaster(id);
        if (b!= null && b.getScope() == Broadcaster.SCOPE.REQUEST){
            throw new IllegalStateException("Broadcaster " + b
                    + " cannot be looked up as its scope is REQUEST");
        }

        if (b!= null && !c.isAssignableFrom(b.getClass())){
            String em = "Invalid BroadcasterLookup class " + c.getName() + ". Cached class is: " + b.getClass().getName();
            LoggerUtils.getLogger().log(Level.WARNING,em);
            throw new IllegalStateException(em);
        }

        return b;
    }

    /**
     * Return a {@link Broadcaster} based on its name.
     * @param name The unique ID
     * @return a {@link Broadcaster}, or null
     */
    static Broadcaster getBroadcaster(Object name){
        return store.get(name);
    }

    /**
     * Add a {@link Broadcaster} to the list.
     * @param b a {@link Broadcaster}
     * @return false if a with the same name {@link Broadcaster} was already stored
     */
    public final static boolean add(Broadcaster b,Object id){
        return (store.putIfAbsent(id,b) == null ? true: false);
    }

    /**
     * Remove a {@link Broadcaster} to the list.
     * @param b a {@link Broadcaster}
     * @return false if wasn't present, or {@link Broadcaster}
     */
    public final static boolean remove(Broadcaster b,Object id){
        return (store.remove(id) == null ? false : true);
    }

    /**
     * Create a new instance of {@link Broadcaster} and store it for
     *
     * @param c The {@link Broadcaster} class instance.
     * @param id The unique ID used to retrieve {@link Broadcaster}
     * @return a new instance of {@link Broadcaster}
     */
    public final static Broadcaster build(Class<? extends Broadcaster> c, Object id){
        Broadcaster b  = null;
        try {
            b = c.newInstance();
        } catch (InstantiationException ex) {
            LoggerUtils.getLogger().log(Level.SEVERE, null, ex);
        } catch (IllegalAccessException ex) {
            LoggerUtils.getLogger().log(Level.SEVERE, null, ex);
        }

        if (id instanceof String){
            b.setID(id.toString());
        }
        store.put(id, b);
        return b;
    }

    /**
     * Shutdown all {@link Broadcaster}
     */
    public final static void destroy(){
        Enumeration<Broadcaster> e = store.elements();
        while (e.hasMoreElements()){
            e.nextElement().destroy();
        }
    }
}
