/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.container;

import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.atmosphere.cpr.AsynchronousProcessor;
import org.atmosphere.cpr.AtmosphereResourceImpl;
import org.atmosphere.cpr.AtmosphereServlet;
import org.atmosphere.cpr.AtmosphereServlet.Action;
import org.atmosphere.cpr.AtmosphereServlet.AtmosphereConfig;
import org.atmosphere.cpr.Broadcaster;
import org.atmosphere.cpr.CometSupport;

/**
 * This class gets used when the {@link AtmosphereServlet} fails to autodetect
 * the Servlet Container we are running on.
 * 
 * This {@link CometSupport} implementation uses a blocking approach, meaning
 * the request thread will be blocked until another Thread invoke the 
 * {@link Broadcaster#broadcast}
 * 
 * @author Jeanfrancois Arcand
 */
public class BlockingIOCometSupport extends AsynchronousProcessor implements CometSupport<AtmosphereResourceImpl> {

    protected final static String LATCH = "org.atmosphere.container.BlockingIOCometSupport.latch";

    protected ConcurrentHashMap<Integer,CountDownLatch> latchs
            = new ConcurrentHashMap<Integer, CountDownLatch>();


    public BlockingIOCometSupport(AtmosphereConfig config){
        super(config);
    }

    /**
     * {@inheritDoc}
     */
    public Action service(HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {

        Action action = suspended(req, res);
        if (action.type == Action.TYPE.SUSPEND) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Suspending" + res);
            }
            suspend(action, req, res);
        } else if (action.type == Action.TYPE.RESUME) {
            if (logger.isLoggable(Level.FINE)) {
                logger.fine("Resuming" + res);
            }

            int latchId = (Integer) req.getSession().
                    getAttribute(LATCH);
            CountDownLatch latch = latchs.get(latchId);

            if (latch == null){
                logger.fine("That response " + res + " wasn't suspended.");
                return action;
            }

            latch.countDown();

            Action nextAction = resumed(req, res);
            if (nextAction.type == Action.TYPE.SUSPEND) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Suspending after Resuming" + res);
                    suspend(action, req, res);
                }
            }
        }
        return action;
    }

    /**
     * Suspend the connection by blocking the current {@link Thread}
     * @param action The {@link AtmosphereServlet.Action}
     * @param req the {@link HttpServletRequest}
     * @param res the {@link HttpServletResponse}
     */
    protected void suspend(Action action, HttpServletRequest req, HttpServletResponse res)
            throws IOException, ServletException {
        CountDownLatch latch = new CountDownLatch(1);

        int hash = latch.hashCode();
        req.setAttribute(LATCH, hash);
        latchs.put(hash, latch);

        if (supportSession()){
            // Store as well in the session in case the resume operation
            // happens outside the AtmosphereHandler.onStateChange scope.
            req.getSession().setAttribute(LATCH, hash);
        }
        try {
            if (action.timeout != -1) {
                latch.await(action.timeout, TimeUnit.MILLISECONDS);
            } else {
                latch.await();
            }            
        } catch (InterruptedException ex) {
        } finally {
            timedout(req, res);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void action(AtmosphereResourceImpl actionEvent)  {
        super.action(actionEvent);
        try {
            if (actionEvent.action().type == Action.TYPE.RESUME) {
                int latchId = -1;
                
                if (actionEvent.getRequest().getAttribute(LATCH) != null){
                    latchId = (Integer) actionEvent.getRequest().getAttribute(LATCH);
                }
                
                if (latchId != -1 && supportSession()){
                    if (actionEvent.getRequest().getSession().getAttribute(LATCH) != null){
                        latchId = (Integer)  actionEvent.getRequest().getSession()
                            .getAttribute(LATCH);
                    }
                }

                if (latchId != -1){
                    CountDownLatch latch = latchs.remove(latchId);
                    latch.countDown();
                } else {
                    logger.log(Level.SEVERE, "Unable to resume the suspended connection");
                }
            }
        } catch (Exception ex) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "", ex);
            }
        }
    }
}
