/*
 * 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */
package org.atmosphere.cpr;

import java.io.IOException;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import org.atmosphere.cpr.Broadcaster.SCOPE;
import org.atmosphere.util.BroadcasterLookup;
import org.atmosphere.util.LoggerUtils;

/**
 * {@link Broadcaster} implementation.
 *
 * Broadcast messages to suspended reponse using the caller's Thread.
 * This basic {@link Broadcaster} isn't using any {@link ExecutorService} to
 * broadcast messages, so it is single threaded. 
 * 
 * @author Jeanfrancois Arcand
 */
public class DefaultBroadcaster implements Broadcaster {

    protected final ConcurrentLinkedQueue<AtmosphereResource> events =
            new ConcurrentLinkedQueue<AtmosphereResource>();
    protected BroadcasterConfig bc;
    protected final BlockingQueue<Entry> messages =
            new LinkedBlockingQueue<Entry>();
    protected final AtomicBoolean started = new AtomicBoolean(false);
    protected SCOPE scope = SCOPE.APPLICATION;
    protected String name = DefaultBroadcaster.class.getSimpleName();
    protected final ConcurrentLinkedQueue<Entry> delayedBroadcast =
            new ConcurrentLinkedQueue<Entry>();
    protected final ConcurrentLinkedQueue<BroadcasterStateListener> listeners =
            new ConcurrentLinkedQueue<BroadcasterStateListener>();

    public DefaultBroadcaster() {
        bc = new BroadcasterConfig();
        //This is ugly, but a better solution will eventually comes.
        (new BroadcasterLookup()).add(this);
    }

    /**
     * {@inheritDoc}
     */
    public void destroy() {
        bc.destroy();
    }

    /**
     * {@inheritDoc}
     */
    public Iterator<AtmosphereResource> getAtmosphereResources() {
        return events.iterator();
    }

    /**
     * {@inheritDoc}
     */
    public void setScope(SCOPE scope) {
        this.scope = scope;
        updateListeners(CHANGE.UPDATED,scope);
    }

    /**
     * {@inheritDoc}
     */
    public SCOPE getScope() {
        return scope;
    }

    /**
     * {@inheritDoc}
     */
    public void setName(String name) {
        updateListeners(CHANGE.REMOVED,this);
        this.name = name;
        updateListeners(CHANGE.ADDED,this);
    }

    /**
     * {@inheritDoc}
     */
    public String getName() {
        return name;
    }

    public class Entry {

        public Object message;
        public Object eventsToPush;
        public BroadcasterFuture f;

        public Entry(Object message, Object eventsToPush, BroadcasterFuture f) {
            this.message = message;
            this.eventsToPush = eventsToPush;
            this.f = f;
        }
    }

    protected void start() {
        if (!started.getAndSet(true)) {
            bc.getExecutorService().submit(new Runnable() {

                public void run() {
                    Entry msg = null;
                    try {
                        msg = messages.take();
                        // Leader/follower
                        bc.getExecutorService().submit(this);

                        if (!delayedBroadcast.isEmpty()) {
                            Iterator<Entry> i = delayedBroadcast.iterator();
                            while (i.hasNext()) {
                                Entry e = i.next();
                                try {
                                    push(e);
                                } finally {
                                    i.remove();
                                    e.f.done();
                                }
                            }
                        }
                        push(msg);
                    } catch (InterruptedException ex) {
                        LoggerUtils.getLogger().log(Level.SEVERE, null, ex);
                    } finally {
                        if (msg != null) {
                            msg.f.done();
                        }
                    }
                }
            });
        }
    }

    protected void push(Entry msg) {
        AtmosphereResourceEventImpl e = null;
        if (msg.eventsToPush == null) {
            for (AtmosphereResource r : events) {
                synchronized (r) {
                    if (r instanceof AtmosphereResourceImpl) {
                        e = ((AtmosphereResourceImpl)r).event();
                        e.setMessage(msg.message);
                    }
                    broadcast(r,e);
                }
            }
        } else if (msg.eventsToPush instanceof AtmosphereResource) {
            AtmosphereResource r = (AtmosphereResource) msg.eventsToPush;
            synchronized (r) {
                if (r instanceof AtmosphereResourceImpl) {
                    e = ((AtmosphereResourceImpl)r).event();
                    e.setMessage(msg.message);
                }
                broadcast(r,e);
            }
        } else if (msg.eventsToPush instanceof Set) {
            Set<AtmosphereResource> sub =
                    (Set<AtmosphereResource>) msg.eventsToPush;
            for (AtmosphereResource r : sub) {
                synchronized (r) {
                    if (r instanceof AtmosphereResourceImpl) {
                        e = ((AtmosphereResourceImpl)r).event();
                        e.setMessage(msg.message);
                    }
                    broadcast(r,e);
                }
            }
        }
    }

    protected void broadcast(final AtmosphereResource r,final AtmosphereResourceEvent e) {
        try {
            r.getAtmosphereConfig().getAtmosphereHandler().onStateChange(e);
        } catch (IOException ex) {
            // AtmosphereHandler.onException?
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
        } catch (IllegalStateException ex) {
            LoggerUtils.getLogger().log(Level.WARNING, "", ex);
            removeAtmosphereResource(r);
        }
    }

    /**
     * {@inheritDoc}
     */
    public BroadcasterFuture<Object> broadcast(Object msg) {
        start();
        msg = bc.filter(msg);
        if (msg == null) {
            return null;
        }
        BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(msg);
        messages.offer(new Entry(msg, null, f));
        return f;
    }

    /**
     * {@inheritDoc}
     */
    public BroadcasterFuture<Object> broadcast(Object msg, AtmosphereResource r) {
        start();
        msg = (String) bc.filter(msg);
        if (msg == null) {
            return null;
        }
        BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(msg);
        messages.offer(new Entry(msg, r, f));
        return f;
    }

    /**
     * {@inheritDoc}
     */
    public BroadcasterFuture<Object> broadcast(Object msg, Set<AtmosphereResource> subset) {
        start();
        msg = (String) bc.filter(msg);
        if (msg == null) {
            return null;
        }
        BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(msg);
        messages.offer(new Entry(msg, subset, f));
        return f;
    }

    /**
     * {@inheritDoc}
     */
    public AtmosphereResource addAtmosphereResource(AtmosphereResource r) {
        if (events.contains(r)) {
            return null;
        }
        updateListeners(CHANGE.ADDED,r);
        events.offer(r);
        return r;
    }

    /**
     * {@inheritDoc}
     */
    public AtmosphereResource removeAtmosphereResource(AtmosphereResource r) {
        if (!events.contains(r)) {
            return null;
        }
        updateListeners(CHANGE.REMOVED,r);
        events.remove(r);
        return r;
    }

    /**
     * Return the list of AtmosphereResource registered with this Broadcaster
     * @return
     */
    protected ConcurrentLinkedQueue<AtmosphereResource> atmosphereResources() {
        return events;
    }

    /**
     * Set the {@link BroadcasterConfig} instance.
     * @param bc
     */
    public void setBroadcasterConfig(BroadcasterConfig bc) {
        this.bc = bc;
    }

    /**
     * Return the current {@link BroadcasterConfig}
     * @return the current {@link BroadcasterConfig}
     */
    public BroadcasterConfig getBroadcasterConfig() {
        return bc;
    }

    /**
     * {@inheritDoc}
     */
    public Future<Object> delayBroadcast(Object o) {
        return delayBroadcast(o, 0, null);
    }

    /**
     * {@inheritDoc}
     */
    public Future<Object> delayBroadcast(final Object o, long delay, TimeUnit t) {
        if (t == null) {
            BroadcasterFuture<Object> f = new BroadcasterFuture<Object>(o);
            delayedBroadcast.offer(new Entry(o, null, f));
            return f;
        } else {
            return bc.getScheduledExecutorService().schedule(new Callable<Object>() {

                public Object call() throws Exception {
                    broadcast(o);
                    return o;
                }
            }, delay, t);
        }
    }

    /**
     * {@inheritDoc}
     */
    public Future<?> scheduleFixedBroadcast(final Object o, long period, TimeUnit t) {
        if (period == 0 || t == null) {
            return null;
        }

        return bc.getScheduledExecutorService().scheduleAtFixedRate(new Runnable() {

            public void run() {
                broadcast(o);
            }
        }, 0, period, t);
    }

    protected void updateListeners(CHANGE c,Object o){
        for (BroadcasterStateListener l : listeners){
            l.changed(c,o);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void addBroadcasterStateListener(BroadcasterStateListener bl) {
        if (listeners.contains(bl)) return;
        listeners.add(bl);
    }

    /**
     * {@inheritDoc}
     */
    public void removeBroadcasterStateListener(BroadcasterStateListener bl) {
        listeners.remove(bl);
    }
}
