/*
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2007-2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 */

package org.atmosphere.util;


import java.util.concurrent.ConcurrentHashMap;
import org.atmosphere.cpr.Broadcaster;
import org.atmosphere.cpr.Broadcaster.CHANGE;

/**
 * A class that can be used to lookup {@link Broadcaster}
 * based on the returned value of {@link Broadcaster#getName()} and
 * {@link Broadcaster#getScope()}
 *
 * @author Jeanfrancois Arcand
 */
public class BroadcasterLookup implements Broadcaster.BroadcasterStateListener {

    private final static ConcurrentHashMap<String,Broadcaster> store
            = new ConcurrentHashMap<String,Broadcaster>();

    /**
     * Lookup a {@link Broadcaster} instance using {@link Broadcaster#getName()}.
     * @param name The Broadcaster's name.
     * @return a Broadcaster, or null if not found.
     * @throws OutOfScopeException
     */
    public final static Broadcaster lookup(String name) throws OutOfScopeException{
        Broadcaster b = getBroadcaster(name);
        if (b!= null && b.getScope() == Broadcaster.SCOPE.REQUEST){
            throw new OutOfScopeException(b);
        }
        return b;
    }

    /**
     * Return a {@link Broadcaster} based on its name.
     * @param name
     * @return a {@link Broadcaster}, or null
     */
    static Broadcaster getBroadcaster(String name){
        return store.get(name);
    }

    /**
     * Add a {@link Broadcaster} to the list.
     * @param b a {@link Broadcaster}
     * @return false if a with the same name {@link Broadcaster} was already stored
     */
    public final boolean add(Broadcaster b){
        b.addBroadcasterStateListener(this);
        return (store.putIfAbsent(b.getName(),b) == null ? true: false);
    }

    /**
     * Remove a {@link Broadcaster} to the list.
     * @param b a {@link Broadcaster}
     * @return false if wasn't present, or {@link Broadcaster}
     */
    public final boolean remove(Broadcaster b){
        b.removeBroadcasterStateListener(this);
        return (store.remove(b.getName()) == null ? false : true);
    }

    public void changed(CHANGE c, Object o) {
        switch (c){
            case ADDED:
                if (o instanceof Broadcaster){
                    Broadcaster b = (Broadcaster)o;
                    store.put(b.getName(), b);
                }
                break;
            case REMOVED:
                if (o instanceof Broadcaster){
                    Broadcaster b = (Broadcaster)o;
                    store.remove(b.getName());
                }
                break;
            default:
                break;
        }
    }

    static public class OutOfScopeException extends Exception{
        public OutOfScopeException(Broadcaster b){
            super("Broadcaster " + b.getName() + " cannot be used. Out of scope");
        }

    }
}
