/*
 * Copyright 2014 The Vibe Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.atmosphere.vibe.server;

import org.atmosphere.vibe.platform.Action;
import org.atmosphere.vibe.platform.server.ServerHttpExchange;
import org.atmosphere.vibe.platform.server.ServerWebSocket;

/**
 * Interface used to interact with the socket.
 * <p>
 * A {@code Server} instance provides {@link ServerSocket} processing HTTP request and
 * WebSocket under the specific URI pattern and manages their life cycles. The
 * {@code Server} API is used to accept socket and to find socket by id and tag.
 * If you are using dependency injection support, make a {@code Server} as
 * component and inject it wherever you need to handle socket.
 * <p>
 * The {@code Server} is a wes application so can be installed on any platform
 * like Servlet wes supports. For that reason, {@code Server} doesn't concern
 * I/O details and I/O details should be configured in the platform following
 * its policy.
 * <p>
 * Server may be accessed by multiple threads.
 * 
 * @author Donghwan Kim
 * @see <a
 *      href="https://github.com/vibe-project/vibe-examples/tree/master/server/platform/"
 *      target="_parent">Examples to install vibe</a>
 */
public interface Server {

    /**
     * Returns a sentence that all of the socket in this server or all of the
     * server if it's in a clustered environment have to follow.
     */
    Sentence all();

    /**
     * Executes the given action retrieving all of the socket in this server or
     * all of the server if it's in a clustered environment .
     */
    Server all(Action<ServerSocket> action);

    /**
     * Returns a sentence that the socket of the given id in this server or all
     * of the server if it's in a clustered environment have to follow.
     */
    Sentence byId(String id);

    /**
     * Executes the given action retrieving the socket of the given id in this
     * server or all of the server if it's in a clustered environment. The given
     * action will be executed only once if socket is found and won't be
     * executed if not found.
     */
    Server byId(String id, Action<ServerSocket> action);

    /**
     * Returns a sentence that the socket tagged with all of the given names in
     * this server or all of the server if it's in a clustered environment have
     * to follow.
     */
    Sentence byTag(String... names);

    /**
     * Executes the given action retrieving the socket tagged with the given
     * name in this server or all of the server if it's in a clustered
     * environment. The given action will be executed multiple times if sockets
     * are found and won't be executed if not found.
     */
    Server byTag(String name, Action<ServerSocket> action);

    /**
     * Executes the given action retrieving the socket tagged with all of the
     * given names in this server or all of the server if it's in a clustered
     * environment. The given action will be executed multiple times if sockets
     * are found and won't be executed if not found.
     */
    Server byTag(String[] names, Action<ServerSocket> action);

    /**
     * Registers an action to be called when the socket has been opened in this
     * server regardless of clustering. It's allowed to add several actions
     * before and after installation, so you don't need to centralize all your
     * code to one class.
     */
    Server socketAction(Action<ServerSocket> action);

    /**
     * ServerHttpExchange action to install in wes
     */
    Action<ServerHttpExchange> httpAction();

    /**
     * ServerWebSocket action to install in wes
     */
    Action<ServerWebSocket> websocketAction();

}