package org.atnos.eff

import cats._, data._
import cats.implicits._
import org.atnos.eff.all._
import Interpret._

/**
 * Effect for computation which can fail but will accumulate errors
 *
 * The runValidate interpreter just collects the messages and returns them at the end
 *
 */
trait ValidateEffect extends
  ValidateCreation with
  ValidateInterpretation

object ValidateEffect extends ValidateEffect

sealed trait Validate[+E, A]
case class Correct[E]() extends Validate[E, Unit]
case class Wrong[E](e: E) extends Validate[E, Unit]

trait ValidateCreation {

  /** create an Validate effect from a single Option value */
  def validateOption[R, E, A](option: Option[A], e: E)(implicit m: Validate[E, ?] |= R): Eff[R, Unit] =
    option.map(_ => correct(())).getOrElse(wrong(e))

  /** create an Validate effect from a single Xor value */
  def validateXor[R, E, A](xor: E Xor A)(implicit m: Validate[E, ?] |= R): Eff[R, Unit] =
    xor.fold(e => wrong(e), _ => correct(()))

  /** create a failed value */
  def wrong[R, E](e: E)(implicit m: Validate[E, ?] |= R): Eff[R, Unit] =
    send[Validate[E, ?], R, Unit](Wrong(e))

  /** create a correct value */
  def correct[R, E, A](a: A)(implicit m: Validate[E, ?] |= R): Eff[R, A] =
    send[Validate[E, ?], R, Unit](Correct[E]()) >> Eff.EffMonad[R].pure(a)

  /** check a correct condition */
  def validateCheck[R, E](condition: Boolean, e: E)(implicit m: Validate[E, ?] |= R): Eff[R, Unit] =
    if (condition) correct(()) else wrong(e)

  /** check a correct value */
  def validateValue[R, E, A](condition: Boolean, a: A, e: E)(implicit m: Validate[E, ?] |= R): Eff[R, A] =
    if (condition) correct(a) else wrong(e) >> Eff.EffMonad[R].pure(a)
}

object ValidateCreation extends ValidateCreation

trait ValidateInterpretation extends ValidateCreation {

  /** run the validate effect, yielding a ValidatedNel */
  def runValidatedNel[R, U, E, A](r: Eff[R, A])(implicit m: Member.Aux[Validate[E, ?], R, U]): Eff[U, ValidatedNel[E, A]] =
    runNel(r).map(result => Validated.fromEither(result.toEither))

  /** run the validate effect, yielding a non-empty list of failures Xor A */
  def runNel[R, U, E, A](r: Eff[R, A])(implicit m: Member.Aux[Validate[E, ?], R, U]): Eff[U, NonEmptyList[E] Xor A] =
    runMap[R, U, E, NonEmptyList[E], A](r)((e: E) => NonEmptyList.of(e))

  /** run the validate effect, yielding a list of failures Xor A */
  def runMap[R, U, E, L : Semigroup, A](r: Eff[R, A])(map: E => L)(implicit m: Member.Aux[Validate[E, ?], R, U]): Eff[U, L Xor A] = {
    val recurse: StateRecurse[Validate[E, ?], A, L Xor A] = new StateRecurse[Validate[E, ?], A, L Xor A] {
      type S = Option[L]
      val init: Option[L] = None

      def apply[X](x: Validate[E, X], s: Option[L]): (X, Option[L]) =
        x match {
          case Wrong(e) => ((), s.fold(Option(map(e)))(l => Option(l |+| map(e))))
          case Correct() => ((), s)
        }

      def applicative[X](xs: List[Validate[E, X]], s: S): (List[X], S) Xor (Validate[E, List[X]], S) = {
        val (state, elements) = xs.foldLeft((s, Vector.empty[X])) { case ((state, list), cur) =>
          cur match {
            case Correct() => (state, list :+ ().asInstanceOf[X])
            case Wrong(e)  => (state.map(_ |+| map(e)) , list :+ ().asInstanceOf[X])
          }
        }
        Xor.Left((elements.toList, state))
      }

      def finalize(a: A, s: S): L Xor A =
        s.fold(Xor.right[L, A](a))(Xor.left[L, A])
    }

    interpretState1[R, U, Validate[E, ?], A, L Xor A]((a: A) => Xor.right[L, A](a))(recurse)(r)
  }

  /** catch and handle possible wrong values */
  def catchWrong[R, E, A](r: Eff[R, A])(handle: E => Eff[R, A])(implicit member: (Validate[E, ?]) <= R): Eff[R, A] = {
    val loop = new StatelessLoop[Validate[E,?], R, A, Eff[R, A]] {
      def onPure(a: A): Eff[R, A] Xor Eff[R, A] =
        Xor.Right(pure(a))

      def onEffect[X](m: Validate[E, X], continuation: Arrs[R, X, A]): Eff[R, A] Xor Eff[R, A] =
        m match {
          case Correct() => Xor.Left(continuation(()))
          case Wrong(e)  => Xor.Left(handle(e))
        }

      def onApplicativeEffect[X](xs: List[Validate[E, X]], continuation: Arrs[R, List[X], A]): Eff[R, A] Xor Eff[R, A] = {
        val issues: List[E] = xs.collect { case Wrong(e) => e }.asInstanceOf[List[E]] // why is asInstanceOf necessary here?

        issues match {
          case Nil => Xor.Left(continuation(List.fill(xs.size)(().asInstanceOf[X])))
          case is  => Xor.Left(EffApplicative.traverse(is)(handle).map(_.head))
        }
      }

    }

    interceptStatelessLoop[R, Validate[E,?], A, A]((a: A) => pure(a), loop)(r)
  }
}

object ValidateInterpretation extends ValidateInterpretation

