package org.atnos.eff

import java.util
import java.util.function._
import java.util.concurrent.ConcurrentHashMap
import cats.Eval

/**
 * This cache is used to memoize a sequence of values for a given key
 *
 * When the cache is "closed" all subsequent calls to "append" will return
 * previously stored values until the last value is reached.
 *
 * This cache is used to memoize a sequence of effects leading to the computation of an Eff[R, A]
 *
 * For example:
 *
 *  val action = asyncFork(1 + 1).flatMap(_ => asyncFork(2 + 2)).flatMap(_ => asyncFork(3 + 3))
 *
 * if this Async action is memoized then the successive computations 1+1, 2+2, 3+3 will not be recomputed
 */
trait SequenceCache extends Cache {

  /**
   * append a new value for the given key
   * when the cache is closed return previous values
   */
  def append[V](key: AnyRef, value: =>V): V

  /**
   * close the given key, the passed value is not cached and just returned for convenience
   */
  def close[V](key: AnyRef): SequenceCache

  /**
   * A SequenceCache can be used as a normal Cache
   * by closing the sequence after the first value has been added
   */
  def memo[V](key: AnyRef, value: =>V): V = {
    val v = append(key, value)
    close(key)
    v
  }
}

/**
 * type class for effects which can be cached
 * in a SequenceCache
 */
trait SequenceCached[M[_]] {
  def apply[X](cache: SequenceCache, key: AnyRef, tx: M[X]): M[X]
}

/**
 * Implementation of a SequenceCache with a concurrent hashmap (unbounded, be careful!)
 */
case class ConcurrentHashMapSequenceCache(map: ConcurrentHashMap[Int, MemoizedSequence] = new ConcurrentHashMap[Int, MemoizedSequence]) extends SequenceCache {

  def append[V](key: AnyRef, value: =>V): V = {
    lazy val v = value

    map.computeIfPresent(key.hashCode,
      new BiFunction[Int, MemoizedSequence, MemoizedSequence] {
        def apply(k: Int, seq: MemoizedSequence) =
          if (seq.isClosed) seq.dequeue else seq.append(v)
      }
    )

    map.computeIfAbsent(key.hashCode,
      new Function[Int, MemoizedSequence] {
        def apply(k: Int) = OpenMemoizedSequence().append(v)
      }
    ).headOption.map(_.value.asInstanceOf[V]).getOrElse(v)

  }

  def close[V](key: AnyRef): SequenceCache = {
    map.computeIfPresent(key.hashCode,
      new BiFunction[Int, MemoizedSequence, MemoizedSequence] {
        def apply(k: Int, seq: MemoizedSequence) = seq.close
      })

    map.computeIfAbsent(key.hashCode,
      new Function[Int, MemoizedSequence] {
        def apply(k: Int) = OpenMemoizedSequence().close
      }
    )

    this
  }

}

sealed trait MemoizedSequence {
  def append(a: =>Any): MemoizedSequence
  def dequeue: MemoizedSequence
  def close: MemoizedSequence
  def isClosed: Boolean
  def headOption: Option[Eval[Any]]
}

case class OpenMemoizedSequence(values: List[Eval[Any]] = Nil) extends MemoizedSequence {

  def append(a: =>Any): MemoizedSequence =
    copy(Eval.later(a).memoize :: values)

  def dequeue: MemoizedSequence =
    this

  def close: MemoizedSequence =
    ClosedMemoizedSequence(None, values.reverse)

  def isClosed: Boolean =
   false

  def headOption: Option[Eval[Any]] =
    values.headOption

  override def toString =
    s"OpenMemoizedSequence(${values.map(_.value)})"
}

case class ClosedMemoizedSequence(headOption: Option[Eval[Any]], tail: List[Eval[Any]] = Nil) extends MemoizedSequence {
  def append(a: =>Any): MemoizedSequence = this

  def close: MemoizedSequence = this

  def isClosed: Boolean =
    true

  def dequeue: MemoizedSequence =
    copy(headOption = tail.headOption.orElse(headOption), tail = tail.drop(1))

  override def toString =
    s"ClosedMemoizedSequence(${(headOption.toList ++ tail).map(_.value)})"

}

/**
 * Implementation of a SequenceCache with a concurrent weak identity hashmap (unbounded, be careful!)
 */
case class ConcurrentWeakIdentityHashMapSequenceCache(
  map: ConcurrentWeakIdentityHashMap[AnyRef, MemoizedSequence] = new ConcurrentWeakIdentityHashMap[AnyRef, MemoizedSequence]) extends SequenceCache {

  def append[V](key: AnyRef, value: =>V): V = {
    lazy val v = value

    map.computeIfPresent(key,
      new BiFunction[AnyRef, MemoizedSequence, MemoizedSequence] {
        def apply(k: AnyRef, seq: MemoizedSequence) =
          if (seq.isClosed) seq.dequeue else seq.append(v)
      }
    )

    map.computeIfAbsent(key,
      new Function[AnyRef, MemoizedSequence] {
        def apply(k: AnyRef) = OpenMemoizedSequence().append(v)
      }
    ).headOption.map(_.value.asInstanceOf[V]).getOrElse(v)

  }

  def close[V](key: AnyRef): SequenceCache = {
    map.computeIfPresent(key,
      new BiFunction[AnyRef, MemoizedSequence, MemoizedSequence] {
        def apply(k: AnyRef, seq: MemoizedSequence) = seq.close
      })

    map.computeIfAbsent(key,
      new Function[AnyRef, MemoizedSequence] {
        def apply(k: AnyRef) = OpenMemoizedSequence().close
      }
    )

    this
  }
}
