package org.atnos.eff

import Eff._
import cats.Eval
import MemoSequenceEffect._

/**
 * Reader effect for memoizing sequences of values
 */
trait MemoSequenceEffect extends
  MemoSequenceTypes with
  MemoSequenceCreation with
  MemoSequenceInterpretation

object MemoSequenceEffect extends MemoSequenceEffect

trait MemoSequenceTypes {
  type _Memoseq[R] = MemoizedSeq <= R
  type _memoseq[R] = MemoizedSeq |= R
}

object MemoSequenceTypes extends MemoSequenceTypes

trait MemoSequenceCreation extends MemoSequenceTypes {

  def append[R :_memoseq, A](key: AnyRef, a: =>A): Eff[R, A] =
    send[MemoizedSeq, R, A](AppendToCache(key, () => a))

  def close[R :_memoseq](key: AnyRef): Eff[R, SequenceCache] =
    send[MemoizedSeq, R, SequenceCache](CloseCache(key))

  def getSequenceCache[R :_memoseq]: Eff[R, SequenceCache] =
    send[MemoizedSeq, R, SequenceCache](GetSequenceCache())

}

trait MemoSequenceInterpretation extends MemoSequenceTypes {

  def runMemoSequence[R, U, A](cache: SequenceCache)(effect: Eff[R, A])(implicit m: Member.Aux[MemoizedSeq, R, U], eval: Eval |= U): Eff[U, A] = {
    interpret.translate(effect)(new Translate[MemoizedSeq  , U] {
      def apply[X](mx: MemoizedSeq[X]): Eff[U, X] =
        mx match {
          case AppendToCache(key, value) => EvalEffect.delay[U, X](cache.append(key, value()))
          case CloseCache(key)           => EvalEffect.delay[U, X](cache.close(key))
          case GetSequenceCache()        => EvalEffect.delay[U, X](cache)
        }
    })
  }

  def runAsyncMemoSequence[R, U, A](cache: SequenceCache)(effect: Eff[R, A])(implicit m: Member.Aux[MemoizedSeq, R, U], async: Async |= U): Eff[U, A] = {
    interpret.translate(effect)(new Translate[MemoizedSeq, U] {
      def apply[X](mx: MemoizedSeq[X]): Eff[U, X] =
        mx match {
          case AppendToCache(key, value) => AsyncEffect.asyncDelay[U, X](cache.append(key, value()))
          case CloseCache(key)           => AsyncEffect.asyncDelay[U, X](cache.close(key))
          case GetSequenceCache()        => AsyncEffect.asyncDelay[U, X](cache)
        }
    })
  }

}

object MemoSequenceInterpretation extends MemoSequenceInterpretation

sealed trait MemoizedSeq[A]

case class AppendToCache[A](key: AnyRef, a: () => A) extends MemoizedSeq[A]
case class CloseCache(key: AnyRef)                   extends MemoizedSeq[SequenceCache]
case class GetSequenceCache()                        extends MemoizedSeq[SequenceCache]
