/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.auth.google.config;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.autumnframework.auth.google.properties.AutumnAuthProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.convert.converter.Converter;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.support.BasicAuthenticationInterceptor;
import org.springframework.security.config.Customizer;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.oauth2.client.CommonOAuth2Provider;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.oauth2.client.oidc.userinfo.OidcUserService;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.oauth2.client.registration.InMemoryClientRegistrationRepository;
import org.springframework.security.oauth2.core.OAuth2AuthenticatedPrincipal;
import org.springframework.security.oauth2.core.user.OAuth2UserAuthority;
import org.springframework.security.oauth2.server.resource.introspection.OAuth2IntrospectionAuthenticatedPrincipal;
import org.springframework.security.oauth2.server.resource.introspection.OAuth2IntrospectionException;
import org.springframework.security.oauth2.server.resource.introspection.OpaqueTokenIntrospector;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestOperations;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.filter.ForwardedHeaderFilter;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Configuration
@EnableWebSecurity
@EnableConfigurationProperties(AutumnAuthProperties.class)
public class SecurityConfig {

    @Autowired
    private AutumnAuthProperties properties;
    
    private ClientRegistrationRepository clientRegistrationRepository;
    
    private OpaqueTokenIntrospector googleIntrospector() {
        return new GoogleIntrospector(properties.getTokenInfoUri(), properties.getUserInfoUri(), 
                properties.getClientid(), properties.getClientsecret());
    }
    
    private ClientRegistration getRegistration() {
        return CommonOAuth2Provider.GOOGLE.getBuilder("google")
                .clientId(properties.getClientid())
                .clientSecret(properties.getClientsecret())
                .build();
    }
    
    private OidcUserService googleUserService() {
        Set<String> googleScopes = new HashSet<>();
        googleScopes.add("https://www.googleapis.com/auth/userinfo.email");
        googleScopes.add("https://www.googleapis.com/auth/userinfo.profile");

        OidcUserService googleUserService = new OidcUserService();
        googleUserService.setAccessibleScopes(googleScopes);
        return googleUserService;
    }

    /**
     * @return a ClientRegistrationRepository
     */
    @Bean
    public ClientRegistrationRepository clientRegistrationRepository() {
        this.clientRegistrationRepository = new InMemoryClientRegistrationRepository(getRegistration());
        return this.clientRegistrationRepository;
    }

    /**
     * @param http Spring Autowired HttpSecurity Object
     * @return a SecurityFilterChain
     * @throws Exception
     */
    @Bean
    public SecurityFilterChain securityFilterChain(HttpSecurity http) throws Exception {
        log.info("Securing all requests for {}, except {}", properties.getAuthorizerequestmatchers(), properties.getIgnoringrequestmatchers());
        
        http
            .authorizeHttpRequests(authorize
                -> authorize
                    .requestMatchers(properties.getAuthorizerequestmatchers()).authenticated()
                        .requestMatchers(properties.getIgnoringrequestmatchers()).permitAll()
                    
            )
            .csrf(csrf -> csrf.ignoringRequestMatchers(properties.getLogouturl()))
            .oauth2Client(Customizer.withDefaults())
            .oauth2Login(oauthLogin
                -> oauthLogin 
                    .userInfoEndpoint(userInfo -> userInfo.oidcUserService(googleUserService()))
                    .loginPage("/oauth2/authorization/google")
            )
            .oauth2ResourceServer(res
                -> res
                    .opaqueToken(token -> token.introspector(googleIntrospector()))
            )
            .logout(logout 
                -> logout
                    .logoutSuccessUrl(properties.getLogoutsuccessurl())
            );
        return http.build();
    }
    
    /**
     * Make sure the headers get forwarded, because the spring boot app properties don't seem to do their job
     * @return The forwarded header filter
     */
    @Bean
    FilterRegistrationBean<ForwardedHeaderFilter> forwardedHeaderFilter() {
        final FilterRegistrationBean<ForwardedHeaderFilter> filterRegistrationBean = new FilterRegistrationBean<ForwardedHeaderFilter>();

        filterRegistrationBean.setFilter(new ForwardedHeaderFilter());
        filterRegistrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE);

        return filterRegistrationBean;
    }
    
    /**
     * Internal class to handle opaque tokens issued by google using the 
     * tokeninfo and userinfo endpoints
     */
    private class GoogleIntrospector implements OpaqueTokenIntrospector {

        private final RestOperations restOperations;

        private Converter<String, RequestEntity<?>> introspectionRequestEntityConverter;
        
        private Converter<String, RequestEntity<?>> userInfoRequestEntityConverter;
        
        /**
         * @param introspectionUri The tokeninfo endpoint
         * @param userInfoUri The userinfo endpoint
         * @param clientId the client id
         * @param clientSecret the client secret
         */
        public GoogleIntrospector(String introspectionUri, String userInfoUri, String clientId, String clientSecret) {
            this.introspectionRequestEntityConverter = this.introspectionRequestEntityConverter(URI.create(introspectionUri));
            this.userInfoRequestEntityConverter = this.userInfoRequestEntityConverter(URI.create(userInfoUri));
            RestTemplate restTemplate = new RestTemplate();
            restTemplate.getInterceptors().add(new BasicAuthenticationInterceptor(clientId, clientSecret));
            this.restOperations = restTemplate;
        }
        
        private record GoogleIntrospectionResponse(String azp, String aud, String sub, String scope, String exp,
                String expires_in, String email, String email_verified, String access_type) {
        }
        
        private record GoogleUserInfoResponse(String sub, String name, String given_name, String family_name, 
                String picture, String email, String email_verified, String locale, String hd) {
        }

        @Override
        public OAuth2AuthenticatedPrincipal introspect(String token) {
            // validate token
            RequestEntity<?> requestEntity = this.introspectionRequestEntityConverter.convert(token);
            if (requestEntity == null) {
                throw new OAuth2IntrospectionException("requestEntityConverter returned a null entity");
            }
            GoogleIntrospectionResponse googleIntrospectionResponse = makeRequest(requestEntity).getBody();
            
            Map<String, Object> attributes = new HashMap<>();
            attributes.putAll(getAttributes(googleIntrospectionResponse));

            List<GrantedAuthority> authorities = parseScopeAuthorities(googleIntrospectionResponse.scope,
                                attributes);
            authorities.add(new OAuth2UserAuthority("ROLE_USER", attributes));

            // userinfo
            requestEntity = this.userInfoRequestEntityConverter.convert(token);
            
            GoogleUserInfoResponse googleUserInfoResponse = makeUserInfoRequest(requestEntity).getBody();
            attributes.putAll(getAttributes(googleUserInfoResponse));
            
            return new OAuth2IntrospectionAuthenticatedPrincipal(attributes, authorities);
        }
        
        private Converter<String, RequestEntity<?>> introspectionRequestEntityConverter(URI introspectionUri) {
            return (token) -> {
                HttpHeaders headers = requestHeaders();
                MultiValueMap<String, String> body = requestBody(token);
                return new RequestEntity<>(body, headers, HttpMethod.POST, introspectionUri);
            };
        }
        
        private HttpHeaders requestHeaders() {
            HttpHeaders headers = new HttpHeaders();
            headers.setAccept(Collections.singletonList(MediaType.APPLICATION_JSON));
            return headers;
        }
        
        private MultiValueMap<String, String> requestBody(String token) {
            MultiValueMap<String, String> body = new LinkedMultiValueMap<>();
            body.add("access_token", token);
            return body;
        }

        private ResponseEntity<GoogleIntrospectionResponse> makeRequest(RequestEntity<?> requestEntity) {
            try {
                return this.restOperations.exchange(requestEntity, GoogleIntrospectionResponse.class);
            }
            catch (Exception ex) {
                throw new OAuth2IntrospectionException(ex.getMessage(), ex);
            }
        }
        
        private Converter<String, RequestEntity<?>> userInfoRequestEntityConverter(URI userInfoUri) {
            return (token) -> {
                HttpHeaders headers = requestUserInfoHeaders(token);
                return new RequestEntity<>(null, headers, HttpMethod.GET, userInfoUri);
            };
        }
        
        private HttpHeaders requestUserInfoHeaders(String token) {
            HttpHeaders headers = new HttpHeaders();
            headers.setAccept(Collections.singletonList(MediaType.APPLICATION_JSON));
            headers.setBearerAuth(token);
            return headers;
        }
        
        private ResponseEntity<GoogleUserInfoResponse> makeUserInfoRequest(RequestEntity<?> requestEntity) {
            try {
                return this.restOperations.exchange(requestEntity, GoogleUserInfoResponse.class);
            }
            catch (Exception ex) {
                throw new OAuth2IntrospectionException(ex.getMessage(), ex);
            }
        }

        private Map<String, Object> getAttributes(GoogleIntrospectionResponse googleIntrospectionResponse) {
            return Map.of("aud", googleIntrospectionResponse.aud, "sub", googleIntrospectionResponse.sub, "email",
                    googleIntrospectionResponse.email, "email_verified", googleIntrospectionResponse.email_verified);
        }
        
        private Map<String, Object> getAttributes(GoogleUserInfoResponse googleUserInfoResponse) {
            // may contain nulls and Map.of dpesn't like that
            return new HashMap<>(){
                private static final long serialVersionUID = -2271300289073166974L; {
                put("sub", googleUserInfoResponse.sub);
                put("name", googleUserInfoResponse.name); 
                put("given_name", googleUserInfoResponse.given_name);
                put("family_name", googleUserInfoResponse.family_name);
                put("picture", googleUserInfoResponse.picture);
                put("locale", googleUserInfoResponse.locale);
                put("hd", googleUserInfoResponse.hd);
            }};
        }

        private List<GrantedAuthority> parseScopeAuthorities(String oauth2Scopes, Map<String, Object> attributes) {
            String[] scopes = oauth2Scopes.split(" ");
            List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();

            for (String scope : scopes) {
                authorities.add(new OAuth2UserAuthority("SCOPE_" + scope, attributes));
            }

            return authorities;
        }
    }
}
