/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.jpa.entities;

import static jakarta.persistence.TemporalType.TIMESTAMP;

import java.time.LocalDateTime;

import org.springframework.data.annotation.CreatedDate;
import org.springframework.data.annotation.LastModifiedDate;
import org.springframework.data.jpa.domain.support.AuditingEntityListener;

import jakarta.persistence.Column;
import jakarta.persistence.EntityListeners;
import jakarta.persistence.MappedSuperclass;
import jakarta.persistence.Temporal;

/**
 * @author : Leon Jongsma
 */

import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.autumnframework.service.identifiable.GenericApiIdentifiable;
import org.autumnframework.service.jpa.identifiable.JpaIdentifiable;


/**
 * Provides a base superclass for entities which should have auditable capabilities, providing
 * <p>
 * information on who or what system created each record, when this record was created, who last
 * <p>
 * updated the record and when this update happened.to enhance an entity with auditable capabilities, just extend this class
 * <p>
 * and provide normal jpa entity annotations
 *
 * @author Timon Veenstra [timon at corizon.nl]
 */

@MappedSuperclass
@EntityListeners(AuditingEntityListener.class)
@AllArgsConstructor
@NoArgsConstructor
@Getter
@Setter
public abstract class AbstractBaseEntity implements JpaIdentifiable, GenericApiIdentifiable<Long> {

    private static final long serialVersionUID = 1L;

//    @CreatedBy
//    @Column(nullable = false, updatable = false)
//    protected U createdBy;

    @CreatedDate
    @Temporal(TIMESTAMP)
    @Column(nullable = false, updatable = false)
    protected LocalDateTime creationDate;
//
//    @LastModifiedBy
//    protected U lastModifiedBy;

    @LastModifiedDate
    @Temporal(TIMESTAMP)
    @Column(nullable = false)
    protected LocalDateTime lastModifiedDate;

}