/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.jpa.repositories;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Stream;

import org.hibernate.jpa.HibernateHints;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.jpa.repository.QueryHints;

import jakarta.persistence.QueryHint;
import org.autumnframework.service.jpa.identifiable.JpaApiIdentifiable;


/**
 * A more specific default repository that includes looking up entities by API id
 * @param <T>
 */
public interface JpaApiRepository<T extends JpaApiIdentifiable> extends JpaRepository<T> {

    Optional<T> findByApiId(UUID apiId);

    List<T> findByApiIdIn(Collection<UUID> apiIds);

    @QueryHints({
        @QueryHint(name = HibernateHints.HINT_FETCH_SIZE, value= "1"),
        @QueryHint(name = HibernateHints.HINT_CACHEABLE, value = "false"),
        @QueryHint(name = HibernateHints.HINT_READ_ONLY, value = "true")})
    Stream<T> streamAllByOrderByCreationDate();
    
    @QueryHints({
        @QueryHint(name = HibernateHints.HINT_FETCH_SIZE, value= "1"),
        @QueryHint(name = HibernateHints.HINT_CACHEABLE, value = "false"),
        @QueryHint(name = HibernateHints.HINT_READ_ONLY, value = "true")})
    Stream<T> streamAllByLastModifiedDateAfterOrderByCreationDate(LocalDateTime date);
    
    @QueryHints({
        @QueryHint(name = HibernateHints.HINT_FETCH_SIZE, value= "1"),
        @QueryHint(name = HibernateHints.HINT_CACHEABLE, value = "false"),
        @QueryHint(name = HibernateHints.HINT_READ_ONLY, value = "true")})
    Stream<T> streamAllByLastModifiedDateBetweenOrderByCreationDate(LocalDateTime start, LocalDateTime end);

    Page<T> findAllByLastModifiedDateAfter(LocalDateTime date, Pageable page);
    
    Page<T> findAllByLastModifiedDateBetween(LocalDateTime start, LocalDateTime end, Pageable page);
}
