/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.jpa.services.elementary;

import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;

import org.autumnframework.service.event.metadata.EventMetaData;
import org.autumnframework.service.jpa.identifiable.JpaApiIdentifiable;

/**
 * Delete by api id is a composed service, it exists of services that provide:
 *  - findByApiId
 *  - deleteById
 *
 *  It's composed to use the same implementation, which re-uses code that also handles OnDeleteEvents
 *
 * @param <T>
 */
public interface DeleteByApiIdService<T extends JpaApiIdentifiable> extends ReadFindByApiIdService<T>,
                                                                         DeleteByIdService<T> {

    Logger log = LoggerFactory.getLogger(DeleteByApiIdService.class);

    @Transactional
    default void deleteByApiId(UUID id, EventMetaData eventMetaData) {
        log.trace("deleteByApiId(id: {}, eventMetaData: {})", id, eventMetaData);

        // Using the inherited service calls to obtain and delete entities, this ensures that the standard
        // service logic of the delete by Long id method is also performed for an delete by API id
        this.findByApiId(id).ifPresentOrElse(t -> {
            log.trace("Found by api id {}, delegating delete by id {}", id, t.getId());
            this.delete(t.getId(), eventMetaData);
        }, () -> {
            log.trace("Could not find entity by api id {}, nothing to delete", id);
        });
    }

    default void deleteByApiId(UUID id) {
        UUID newMessageChainId = UUID.randomUUID();

        log.debug("deleteByApiId({}), initiating new message chain id: {}", id, newMessageChainId);

        this.deleteByApiId(id, EventMetaData.builder()
                                            .messageChainId(newMessageChainId)
                                            .build());
    }

}
