/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.jpa.services.genericdefault;

import org.autumnframework.service.identifiable.GenericIdentifiable;
import org.autumnframework.service.event.listeners.generic.GenericOnCreateListener;
import org.autumnframework.service.jpa.services.genericdefault.shared.GenericJpaGetRepositoryService;
import org.autumnframework.service.services.GenericCreateService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

import jakarta.transaction.Transactional;

/**
 * Provides a default implementation for the create service
 * @param <T>
 * @param <ID>
 */
public interface GenericJpaCreateService<T extends GenericIdentifiable<ID>,
                                         ID extends Serializable,
                                         EVENT_METADATA> extends GenericJpaGetRepositoryService<T, ID>,
                                                                 GenericCreateService<T, ID> {

    Logger log = LoggerFactory.getLogger(GenericJpaCreateService.class);

    default List<GenericOnCreateListener<T, ID, EVENT_METADATA>> getOnCreateListeners() {
        log.trace("Returning default: empty OnCreateListener list");
        return Collections.emptyList();
    }

    /**
     * Same as {@link #create(T)}, but offers you to append meta data that is included in the on create event
     * @param t
     * @param eventMetaData
     * @return
     */
    @Transactional
    default T create(T t, EVENT_METADATA eventMetaData) {
        if (log.isTraceEnabled()) {
            log.trace("Create entity: {}", t);
        } else {
            log.debug("Create entity: {}", t.getClass().getSimpleName());
        }

        final T tCreated = this.getRepository().save(t);
        this.getOnCreateListeners().forEach(createEventListener -> {
            if (log.isTraceEnabled()) {
                log.trace("Calling onCreate for {} with id {}, entity: {}, eventMetaData: {}", t.getClass().getSimpleName(), t.getId(), t, eventMetaData);
            } else {
                log.debug("Calling onCreate for {} with id {}", t.getClass().getSimpleName(), t.getId());
            }

            try {
                createEventListener.onCreate(tCreated, eventMetaData);
            } catch (Exception e) {
                log.error("Failed to execute onCreate handler for {} with id {}, continuing with other handlers ", t.getClass().getSimpleName(), t.getId(), e);
            }
        });

        return tCreated;
    }
    
    @Transactional
    default List<T> createAll(List<T> t, EVENT_METADATA eventMetaData) {
        if (!CollectionUtils.isEmpty(t)) {
        
            if (log.isTraceEnabled()) {
                log.trace("Create {} entities: {}", t.size(), t);
            } else {
                log.debug("Create {} entities: {}", t.size(), t.get(0).getClass().getSimpleName());
            }
    
            final List<T> lstCreated = this.getRepository().saveAll(t);
            lstCreated.stream().forEach( tCreated -> {
                this.getOnCreateListeners().forEach(createEventListener -> {
                    if (log.isTraceEnabled()) {
                        log.trace("Calling onCreate for {} with id {}, entity: {}, eventMetaData: {}", t.get(0).getClass().getSimpleName(), t.get(0).getId(), t.get(0), eventMetaData);
                    } else {
                        log.debug("Calling onCreate for {} with id {}", t.getClass().getSimpleName(), t.get(0).getId());
                    }
        
                    try {
                        createEventListener.onCreate(tCreated, eventMetaData);
                    } catch (Exception e) {
                        log.error("Failed to execute onCreate handler for {} with id {}, continuing with other handlers ", t.get(0).getClass().getSimpleName(), t.get(0).getId(), e);
                    }
                });
             });
    
            return lstCreated;
        }
        log.warn("Empty list passed in, nothing persisted");
        return Collections.emptyList();
    }

    default void performOnCreateHandlers() {
        //
    }

    @Override
    default T create(T t) {
        return this.create(t, null);
    }
}
