/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.jpa.services.genericdefault;

import org.autumnframework.service.identifiable.GenericIdentifiable;
import org.autumnframework.service.event.listeners.generic.GenericOnDeleteListener;
import org.autumnframework.service.jpa.services.genericdefault.shared.GenericJpaGetRepositoryService;
import org.autumnframework.service.services.GenericDeleteByIdService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * Provides a default implementation for the delete service
 * @param <ID>
 * @param <T>
 */
public interface GenericJpaDeleteByIdService< T extends GenericIdentifiable<ID>,
                                             ID extends Serializable,
                                             EVENT_METADATA> extends GenericJpaGetRepositoryService<T, ID>,
        GenericDeleteByIdService<T, ID> {

    Logger log = LoggerFactory.getLogger(GenericJpaDeleteByIdService.class);

    default List<GenericOnDeleteListener<T, ID, EVENT_METADATA>> getOnDeleteListeners() {
        log.trace("Returning default: empty OnDeleteListener list");
        return Collections.emptyList();
    }

    /**
     * Same as {@link #deleteById(ID)}, but offers you to append meta data that is included in the on delete event
     * @param id
     * @param eventMetaData
     */
    @Transactional
    default void delete(ID id, EVENT_METADATA eventMetaData) {
        if (eventMetaData != null) {
            log.trace("Delete by id: {}, meta data: {}", id, eventMetaData);
        } else {
            log.trace("Delete by id: {}", id);
        }

        // Only if we have any event listeners, we need to obtain the full entity. Else we just perform the delete
        boolean hasFullEntityReceivingDeleteListeners = !CollectionUtils.isEmpty(getOnDeleteListeners());

        final T t;
        if (hasFullEntityReceivingDeleteListeners) {
            log.trace("Found OnDeleteListeners for entity, looking up entity before deleting it");
            Optional<T> optionalEntity = this.getRepository().findById(id);

            if (optionalEntity.isEmpty()) {
                log.trace("No entity found to delete, nothing to delete.");
                return;
            }

            t = optionalEntity.get();
            log.trace("Found entity: {} with id: {} to delete, caching entity for OnDeleteListener", t.getClass().getSimpleName(), t.getId());
        } else {
            // Required, though never used. Required due to final
            t = null;
        }

        this.getRepository().deleteById(id);

        if (hasFullEntityReceivingDeleteListeners) {
            getOnDeleteListeners().forEach(onDeleteListeners -> {
                if (log.isTraceEnabled()) {
                    log.trace("Calling onDelete for {} with id {}, entity: {}, eventMetaData: {}", t.getClass(), t.getId(), t, eventMetaData);
                } else {
                    log.debug("Calling onDelete for {} with id {}", t.getClass().getSimpleName(), t.getId());
                }

                try {
                    onDeleteListeners.onDelete(t, eventMetaData);
                } catch (Exception e) {
                    log.error("Failed to execute onDelete handler for {} with id {}, continuing with other handlers ", t.getClass().getSimpleName(), t.getId(), e);
                }
            });
        }
    }

    @Override
    @Transactional
    default void deleteById(ID id) {
       this.delete(id, null);
    }

}
