/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.jpa.services.genericdefault;

import org.autumnframework.service.identifiable.GenericIdentifiable;
import org.autumnframework.service.event.listeners.generic.GenericOnUpdateListener;
import org.autumnframework.service.jpa.services.genericdefault.shared.GenericJpaGetRepositoryService;
import org.autumnframework.service.services.GenericUpdateService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

/**
 * Defines a default implementation of update
 */
public interface GenericJpaUpdateService<  T extends GenericIdentifiable<ID>,
                                          ID extends Serializable,
                                          EVENT_METADATA  > extends GenericJpaGetRepositoryService<T, ID>,
                                                                    GenericUpdateService<T, ID> {

    Logger log = LoggerFactory.getLogger(GenericJpaUpdateService.class);

    default List<GenericOnUpdateListener<T, ID, EVENT_METADATA>> getOnUpdateListeners() {
        log.trace("Returning default: empty OnUpdateListener list");
        return Collections.emptyList();
    }

    /**
     * Same as {@link #update(T)}, but allows you to append meta data, which is included in the onUpate event
     * @param t
     * @param eventMetaData
     * @return
     */
    @Transactional
    default T update(T t, EVENT_METADATA eventMetaData) {
        if (log.isTraceEnabled()) {
            log.trace("Update entity: {}", t);
        } else {
            log.debug("Update entity: {} with id: {}", t.getClass().getSimpleName(), t.getId());
        }

        final T tUpdated = this.getRepository().save(t);
        this.getOnUpdateListeners().forEach(onUpdateListener -> {
            if (log.isTraceEnabled()) {
                log.trace("Calling onUpdate for {} with id {}, entity: {}, eventMetaData: {}", t.getClass(), t.getId(), t, eventMetaData);
            } else {
                log.debug("Calling onUpdate for {} with id {}", t.getClass().getSimpleName(), t.getId());
            }

            try {
                onUpdateListener.onUpdate(tUpdated, eventMetaData);
            } catch (Exception e) {
                log.error("Failed to execute onUpdate handler for {} with id {}, continuing with other handlers ", t.getClass().getSimpleName(), t.getId(), e);
            }
        });
        return tUpdated;
    }

    @Transactional
    default List<T> updateAll(List<T> t, EVENT_METADATA eventMetaData) {
        if (!CollectionUtils.isEmpty(t)) {
            if (log.isTraceEnabled()) {
                log.trace("Update {} entities: {}", t.size(), t);
            } else {
                log.debug("Update {} entities: {} with id: {}", t.size(), t.get(0).getClass().getSimpleName(), t.get(0).getId());
            }
    
            final List<T> lstUpdated = this.getRepository().saveAll(t);
            lstUpdated.stream().forEach( tUpdated -> {
                this.getOnUpdateListeners().forEach(onUpdateListener -> {
                    if (log.isTraceEnabled()) {
                        log.trace("Calling onUpdate for {} with id {}, entity: {}, eventMetaData: {}", t.get(0).getClass(), t.get(0).getId(), t.get(0), eventMetaData);
                    } else {
                        log.debug("Calling onUpdate for {} with id {}", t.get(0).getClass().getSimpleName(), t.get(0).getId());
                    }
        
                    try {
                        onUpdateListener.onUpdate(tUpdated, eventMetaData);
                    } catch (Exception e) {
                        log.error("Failed to execute onUpdate handler for {} with id {}, continuing with other handlers ", t.get(0).getClass().getSimpleName(), t.get(0).getId(), e);
                    }
                });
            });
            return lstUpdated;
        }
        log.warn("Empty list passed in, nothing persisted");
        return Collections.emptyList();
    }
    @Override
    @Transactional
    default T update(T t) {
        return this.update(t, null);
    }
}
