/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.pageable;

import lombok.extern.slf4j.Slf4j;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.util.CollectionUtils;
import java.util.Iterator;
import java.util.function.Function;

@Slf4j
public class PagedIterator<T> implements Iterator<T> {
    public static final Pageable PAGEABLE_FINISHED = null;
    private final Function<Pageable, Page<T>> findPaged;
    private Pageable pageable;
    private Page<T> currentPage;
    private Iterator<T> currentPageIterator;

    public PagedIterator(Function<Pageable, Page<T>> findPaged, Pageable pageable) {
        this.findPaged = findPaged;
        this.pageable = pageable;
    }
    @Override
    public boolean hasNext() {
        if (this.pageable == PAGEABLE_FINISHED) {
            log.debug("No pageable, iterator already used once");
            return false;
        }
        if (this.currentPage == null || this.currentPageIterator == null) {
            log.debug("Fetching page {}, requesting {} items", this.pageable.getPageNumber(), this.pageable.getPageSize());
            Page<T> newPage = this.findPaged.apply(this.pageable);
            if (CollectionUtils.isEmpty(newPage.getContent())) {
                log.trace("No content found in page, marking pageable as ended");
                this.pageable = PAGEABLE_FINISHED;
                return false;
            }
            log.trace("Found page number {} of {} pages with {} elements ", newPage.getPageable().getPageNumber(), newPage.getTotalPages(), newPage.getNumberOfElements());
            this.currentPage = newPage;
            this.currentPageIterator = this.currentPage.getContent().iterator();
            return true;
        }
        if (this.currentPageIterator.hasNext()) {
            log.trace("Current page hasNext");
            return true;
        }
        log.trace("Current page {} has no next, selecting next page {}", this.pageable.getPageNumber(), this.pageable.getPageNumber() +1);
        this.pageable = this.pageable.next();
        // Resetting the current page variables, so we can fetch the next
        this.currentPage = null;
        this.currentPageIterator = null;
        return this.hasNext();
    }
    @Override
    public T next() {
        if (this.pageable == PAGEABLE_FINISHED) {
            throw new IllegalStateException("Next called without checking hasNext(). Iterator already used once and ended.");
        } else if (this.currentPage == null || this.currentPageIterator == null) {
            throw new IllegalStateException("Next called without checking hasNext(). No pages fetched yet.");
        }
        return this.currentPageIterator.next();
    }
}
