/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.services;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.stream.Stream;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import org.autumnframework.service.identifiable.GenericApiIdentifiable;

/**
 * In it's bare essentials defines what an audited read findAll service should look like
 * @param <T>
 * @param <ID>
 */
public interface GenericAuditedReadFindAllService< T extends GenericApiIdentifiable<ID>,
                                            ID extends Serializable> {

    Page<T> findAllModifiedSince(@NonNull LocalDateTime date, Pageable page);
    
    Page<T> findAllModifiedBetween(@NonNull LocalDateTime start, @NonNull LocalDateTime end, Pageable page);
    
    Stream<T> streamAllModifiedSince(@NonNull LocalDateTime date);
    
    Stream<T> streamAllModifiedBetween(@NonNull LocalDateTime start, @NonNull LocalDateTime end);
    
    default Page<T> findAllModifiedSince(@NonNull Date date, Pageable page) {
        return this.findAllModifiedSince(LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault()), page);
    }
    
    default Page<T> findAllModifiedBetween(@NonNull Date start, @NonNull Date end, Pageable page) {
        return this.findAllModifiedBetween(LocalDateTime.ofInstant(start.toInstant(), ZoneId.systemDefault()), LocalDateTime.ofInstant(end.toInstant(), ZoneId.systemDefault()), page);
    }
    
    default Stream<T> streamAllModifiedSince(@NonNull Date date) {
        return this.streamAllModifiedSince(LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault()));
    }
    
    default Stream<T> streamAllModifiedBetween(@NonNull Date start, @NonNull Date end) {
        return this.streamAllModifiedBetween(LocalDateTime.ofInstant(start.toInstant(), ZoneId.systemDefault()), LocalDateTime.ofInstant(end.toInstant(), ZoneId.systemDefault()));
    }

}
