/*
 * AvailException.kt
 * Copyright © 1993-2022, The Avail Foundation, LLC.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * * Neither the name of the copyright holder nor the names of the contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package avail.exceptions

import avail.descriptor.numbers.A_Number
import avail.descriptor.representation.AvailObject
import avail.optimizer.jvm.CheckedMethod
import avail.optimizer.jvm.CheckedMethod.Companion.instanceMethod
import avail.optimizer.jvm.ReferencedInGeneratedCode

/**
 * [AvailException] is the root of the hierarchy of [exceptions][Exception] that
 * are specific to the implementation of [AvailObject] and its numerous
 * primitive operations.
 *
 * @property errorCode
 *   The [error value][AvailObject].
 *
 * @constructor
 *   Construct the [AvailException].
 * @param cause
 *   The optional proximal [cause][Throwable] of the [Exception].
 *
 * @author Todd L Smith &lt;todd@availlang.org&gt;
 */
open class AvailException
constructor(
	val errorCode: AvailErrorCode,
	cause: Throwable? = null
): Exception(cause)
{
	/**
	 * Answer the numeric error code as an [Avail][AvailObject].
	 *
	 * @return
	 *   The [numeric&#32;error&#32;code][AvailObject].
	 */
	@ReferencedInGeneratedCode
	fun numericCode(): A_Number = errorCode.numericCode()

	companion object
	{
		/** The [CheckedMethod] for [numericCode]. */
		val numericCodeMethod = instanceMethod(
			AvailException::class.java,
			AvailException::numericCode.name,
			A_Number::class.java)
	}
}
