/*
 * IndexCompressor.kt
 * Copyright © 1993-2022, The Avail Foundation, LLC.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * * Neither the name of the copyright holder nor the names of the contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package avail.serialization

/**
 * This interface defines the abstract protocol for compressing a sequence of
 * integers used by the [Serializer] and [Deserializer].
 */
interface IndexCompressor
{
	/**
	 * Given an index to serialize, or one that was reconstructed during
	 * deserialization, update this compressor to take this index into account,
	 * and answer an encoding that the deserializer will be able to use to get
	 * back to the original index, given a compressor in the same state.
	 */
	fun compress(index: Int): Int

	/**
	 * Given an encoded index that was produced by [compress] from the same
	 * state that the compressor is now in, answer the original [Int] that was
	 * provided to `compress` to get the encoded [Int].  Also ensure the
	 * compressor is updated in the exact same way that the original `compress`
	 * call would have done.
	 */
	fun decompress(index: Int): Int

	/**
	 * Advance the current index number.  This value can be tracked by
	 * subclasses to support current-index-relative (backward) offsets.
	 */
	fun incrementIndex()

	/**
	 * The current index number, which is one more than the last value returned
	 * by [incrementIndex].
	 */
	fun currentIndex(): Int
}
