/*
 * VariableSharedDescriptor.kt
 * Copyright © 1993-2022, The Avail Foundation, LLC.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * * Neither the name of the copyright holder nor the names of the contributors
 *   may be used to endorse or promote products derived from this software
 *   without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package avail.descriptor.variables

import avail.annotations.HideFieldInDebugger
import avail.annotations.HideFieldJustForPrinting
import avail.descriptor.atoms.A_Atom
import avail.descriptor.fiber.A_Fiber.Companion.recordVariableAccess
import avail.descriptor.maps.A_Map.Companion.hasKey
import avail.descriptor.maps.A_Map.Companion.mapAtPuttingCanDestroy
import avail.descriptor.maps.A_Map.Companion.mapSize
import avail.descriptor.maps.A_Map.Companion.mapWithoutKeyCanDestroy
import avail.descriptor.numbers.A_Number
import avail.descriptor.numbers.A_Number.Companion.plusCanDestroy
import avail.descriptor.pojos.RawPojoDescriptor
import avail.descriptor.representation.A_BasicObject
import avail.descriptor.representation.AbstractSlotsEnum
import avail.descriptor.representation.AvailObject
import avail.descriptor.representation.BitField
import avail.descriptor.representation.IntegerSlotsEnum
import avail.descriptor.representation.Mutability
import avail.descriptor.representation.NilDescriptor.Companion.nil
import avail.descriptor.representation.ObjectSlotsEnum
import avail.descriptor.types.A_Type
import avail.descriptor.types.A_Type.Companion.keyType
import avail.descriptor.types.A_Type.Companion.rangeIncludesLong
import avail.descriptor.types.A_Type.Companion.sizeRange
import avail.descriptor.types.A_Type.Companion.valueType
import avail.descriptor.types.A_Type.Companion.writeType
import avail.descriptor.types.TypeTag
import avail.descriptor.types.VariableTypeDescriptor
import avail.descriptor.variables.VariableSharedDescriptor.IntegerSlots.Companion.HASH_ALWAYS_SET
import avail.descriptor.variables.VariableSharedDescriptor.IntegerSlots.HASH_AND_MORE
import avail.descriptor.variables.VariableSharedDescriptor.ObjectSlots.KIND
import avail.descriptor.variables.VariableSharedDescriptor.ObjectSlots.VALUE
import avail.descriptor.variables.VariableSharedDescriptor.ObjectSlots.WRITE_REACTORS
import avail.exceptions.AvailErrorCode.E_CANNOT_READ_UNASSIGNED_VARIABLE
import avail.exceptions.AvailErrorCode.E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE
import avail.exceptions.AvailException
import avail.exceptions.VariableGetException
import avail.exceptions.VariableSetException
import avail.interpreter.execution.AvailLoader
import avail.interpreter.execution.Interpreter
import org.availlang.json.JSONWriter

/**
 * My [object&#32;instances][AvailObject] are [shared][Mutability.SHARED]
 * variables.
 *
 * @author Todd L Smith &lt;todd@availlang.org&gt;
 * @see VariableDescriptor
 *
 * @constructor
 * Construct a new [shared][Mutability.SHARED] [variable][A_Variable].
 *
 * @param mutability
 *   The [mutability][Mutability] of the new descriptor.
 * @param typeTag
 *   The [TypeTag] to embed in the new descriptor.
 * @param objectSlotsEnumClass
 *   The Java [Class] which is a subclass of [ObjectSlotsEnum] and defines this
 *   object's object slots layout, or null if there are no object slots.
 * @param integerSlotsEnumClass
 *   The Java [Class] which is a subclass of [IntegerSlotsEnum] and defines this
 *   object's object slots layout, or null if there are no integer slots.
 */
open class VariableSharedDescriptor protected constructor(
	mutability: Mutability,
	typeTag: TypeTag,
	objectSlotsEnumClass: Class<out ObjectSlotsEnum>?,
	integerSlotsEnumClass: Class<out IntegerSlotsEnum>?
) : VariableDescriptor(
	mutability, typeTag, objectSlotsEnumClass, integerSlotsEnumClass)
{
	/**
	 * The layout of integer slots for my instances.
	 */
	enum class IntegerSlots : IntegerSlotsEnum
	{
		/**
		 * The low 32 bits are used for the hash, but the upper 32 can be used
		 * by subclasses.
		 */
		@HideFieldInDebugger
		HASH_AND_MORE;

		companion object
		{
			/**
			 * A slot to hold the hash value.  Must be computed when (or before)
			 * making a variable shared.
			 */
			val HASH_ALWAYS_SET = BitField(HASH_AND_MORE, 0, 32) { null }

			init
			{
				assert(VariableDescriptor.IntegerSlots.HASH_AND_MORE.ordinal
					== HASH_AND_MORE.ordinal)
				assert(VariableDescriptor.IntegerSlots.HASH_OR_ZERO
					.isSamePlaceAs(HASH_ALWAYS_SET))
			}
		}
	}

	/**
	 * The layout of object slots for my instances.
	 */
	enum class ObjectSlots : ObjectSlotsEnum
	{
		/**
		 * The [contents][AvailObject] of the [variable][VariableDescriptor].
		 */
		VALUE,

		/**
		 * The [kind][AvailObject] of the [variable][VariableDescriptor].  Note
		 * that this is always a [variable&#32;type][VariableTypeDescriptor].
		 */
		KIND,

		/**
		 * A [raw&#32;pojo][RawPojoDescriptor] that wraps a [map][Map] from
		 * arbitrary [Avail&#32;values][AvailObject] to
		 * [write&#32;reactors][VariableDescriptor.VariableAccessReactor] that
		 * respond to writes of the [variable][VariableDescriptor].
		 */
		@HideFieldJustForPrinting
		WRITE_REACTORS;

		companion object
		{
			init
			{
				assert(VariableDescriptor.ObjectSlots.VALUE.ordinal
					== VALUE.ordinal)
				assert(VariableDescriptor.ObjectSlots.KIND.ordinal
					== KIND.ordinal)
				assert(VariableDescriptor.ObjectSlots.WRITE_REACTORS.ordinal
					== WRITE_REACTORS.ordinal)
			}
		}
	}

	override fun allowsImmutableToMutableReferenceInField(
		e: AbstractSlotsEnum
	) = (super.allowsImmutableToMutableReferenceInField(e)
		|| e === VALUE
		|| e === WRITE_REACTORS
		|| e === HASH_AND_MORE) // only for flags.

	override fun o_Hash(self: AvailObject): Int =
		self[HASH_ALWAYS_SET]

	override fun o_Value(self: AvailObject): AvailObject
	{
		recordReadFromSharedVariable(self)
		return self.volatileSlot(VALUE)
	}

	@Throws(VariableGetException::class)
	override fun o_GetValue(self: AvailObject): AvailObject
	{
		recordReadFromSharedVariable(self)
		try
		{
			Interpreter.currentOrNull()?.let { interpreter ->
				if (interpreter.traceVariableReadsBeforeWrites())
				{
					val fiber = interpreter.fiber()
					fiber.recordVariableAccess(self, true)
				}
			}
		}
		catch (e: ClassCastException)
		{
			// No implementation required.
		}
		// Answer the current value of the variable. Fail if no value is
		// currently assigned.
		val value = self.volatileSlot(VALUE)
		if (value.isNil)
		{
			throw VariableGetException(E_CANNOT_READ_UNASSIGNED_VARIABLE)
		}
		assert(value.descriptor().isShared)
		return value
	}

	@Throws(VariableGetException::class)
	override fun o_GetValueClearing(self: AvailObject): AvailObject
	{
		recordReadFromSharedVariable(self)
		try
		{
			Interpreter.currentOrNull()?.let { interpreter ->
				if (interpreter.traceVariableReadsBeforeWrites())
				{
					val fiber = interpreter.fiber()
					fiber.recordVariableAccess(self, true)
				}
			}
		}
		catch (e: ClassCastException)
		{
			// No implementation required.
		}
		// Answer the current value of the variable. Fail if no value is
		// currently assigned.
		val value = self.volatileSlot(VALUE)
		if (value.isNil)
		{
			throw VariableGetException(E_CANNOT_READ_UNASSIGNED_VARIABLE)
		}
		assert(value.descriptor().isShared)
		handleVariableWriteTracing(self)
		self.setVolatileSlot(VALUE, nil)
		recordWriteToSharedVariable()
		return value
	}

	override fun o_HasValue(self: AvailObject): Boolean
	{
		recordReadFromSharedVariable(self)
		try
		{
			Interpreter.currentOrNull()?.let { interpreter ->
				if (interpreter.traceVariableReadsBeforeWrites())
				{
					val fiber = interpreter.fiber()
					fiber.recordVariableAccess(self, true)
				}
			}
		}
		catch (e: ClassCastException)
		{
			// No implementation required.
		}
		return self.volatileSlot(VALUE).notNil
	}

	@Throws(VariableSetException::class)
	override fun o_SetValue(self: AvailObject, newValue: A_BasicObject)
	{
		val outerKind: A_Type = self[KIND]
		if (!newValue.isInstanceOf(outerKind.writeType))
		{
			throw VariableSetException(E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
		}
		handleVariableWriteTracing(self)
		self.setVolatileSlot(VALUE, newValue.makeShared())
		recordWriteToSharedVariable()
	}

	@Throws(VariableSetException::class)
	override fun o_SetValueNoCheck(
		self: AvailObject,
		newValue: A_BasicObject)
	{
		assert(newValue.notNil)
		handleVariableWriteTracing(self)
		self.setVolatileSlot(VALUE, newValue.makeShared())
		recordWriteToSharedVariable()
	}

	@Throws(VariableGetException::class, VariableSetException::class)
	override fun o_GetAndSetValue(
		self: AvailObject, newValue: A_BasicObject): AvailObject
	{
		try
		{
			handleVariableWriteTracing(self)
			val outerKind = self[KIND]
			if (!newValue.isInstanceOf(outerKind.writeType))
			{
				throw VariableSetException(
					E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
			}
			val oldValue = self.getAndSetVolatileSlot(VALUE, newValue)
			if (oldValue.isNil)
			{
				// NOTE: It writes the new value, but still reports an error.
				throw VariableGetException(E_CANNOT_READ_UNASSIGNED_VARIABLE)
			}
			return oldValue
		}
		finally
		{
			recordWriteToSharedVariable()
		}
	}

	@Throws(VariableGetException::class, VariableSetException::class)
	override fun o_CompareAndSwapValues(
		self: AvailObject,
		reference: A_BasicObject,
		newValue: A_BasicObject): Boolean
	{
		if (!newValue.isInstanceOf(self[KIND].writeType))
		{
			throw VariableSetException(E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
		}
		return o_CompareAndSwapValuesNoCheck(self, reference, newValue)
	}

	@Throws(VariableSetException::class)
	override fun o_CompareAndSwapValuesNoCheck(
		self: AvailObject,
		reference: A_BasicObject,
		newValue: A_BasicObject
	): Boolean =
		try
		{
			handleVariableWriteTracing(self)
			// Test if the reference value happens to be semantically equal to
			// the value in the slot.  If so, we use the value from the slot as
			// the reference for the compare-and-set, because it's defined in
			// terms of Kotlin object identity.
			//
			// This doesn't prevent spurious failure from using a reference
			// that's semantically equal but not identical to the value, because
			// the slot can change after we read it to normalize the reference
			// to it (if it's equal).  However, it does prevent *repeated*
			// collisions, and ensures at least somebody is able to write a
			// value using equality for the reference – because the spurious
			// failures can only be a result of some other fiber successfully
			// writing new values to the slot.
			val peek = self.volatileSlot(VALUE)
			val normalizedReference =
				if (reference !== peek && reference.equals(peek)) peek
				else reference
			self.compareAndSetVolatileSlot(
				VALUE, normalizedReference, newValue.makeShared())
		}
		finally
		{
			recordWriteToSharedVariable()
		}

	@Throws(VariableGetException::class, VariableSetException::class)
	override fun o_FetchAndAddValue(
		self: AvailObject,
		addend: A_Number): A_Number
	{
		// Simply read, add, and compare-and-set until it succeeds.
		var oldValue: A_Number
		while (true) {
			oldValue = self.value()
			if (oldValue.isNil)
				throw VariableGetException(E_CANNOT_READ_UNASSIGNED_VARIABLE)
			val newValue = oldValue.plusCanDestroy(addend, false)
			if (o_CompareAndSwapValues(self, oldValue, newValue))
			{
				return oldValue
			}
		}
	}

	@Throws(VariableGetException::class, VariableSetException::class)
	override fun o_AtomicAddToMap(
		self: AvailObject,
		key: A_BasicObject,
		value: A_BasicObject)
	{
		// Simply read, add, and compare-and-set until it succeeds.
		val outerKind: A_Type = self[KIND]
		val writeType = outerKind.writeType
		do
		{
			val oldValue = self.volatileSlot(VALUE)
			if (oldValue.isNil)
				throw VariableGetException(E_CANNOT_READ_UNASSIGNED_VARIABLE)
			if (!oldValue.isMap)
				throw VariableGetException(
					E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
			val newMap = oldValue.mapAtPuttingCanDestroy(key, value, false)
			if (writeType.isMapType)
			{
				// Just check the new size, new key, and new value.
				if (!writeType.sizeRange.rangeIncludesLong(
						newMap.mapSize.toLong())
					|| !key.isInstanceOf(writeType.keyType)
					|| !value.isInstanceOf(writeType.valueType))
				{
					throw VariableSetException(
						E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
				}
			}
			else
			{
				// Do a full type-check.
				if (!newMap.isInstanceOf(writeType))
					throw VariableSetException(
						E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
			}
		}
		while (!o_CompareAndSwapValuesNoCheck(self, oldValue, newMap))
	}

	@Throws(VariableGetException::class, VariableSetException::class)
	override fun o_AtomicRemoveFromMap(
		self: AvailObject,
		key: A_BasicObject)
	{
		// Simply read, remove, and compare-and-set until it succeeds.
		val outerKind: A_Type = self[KIND]
		val writeType = outerKind.writeType
		while (true) {
			val oldValue = self.volatileSlot(VALUE)
			if (oldValue.isNil)
				throw VariableGetException(E_CANNOT_READ_UNASSIGNED_VARIABLE)
			if (!oldValue.isMap)
				throw VariableGetException(
					E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
			val newMap = oldValue.mapWithoutKeyCanDestroy(key, false)
			if (writeType.isMapType)
			{
				// We only have to check the size, since we didn't add any new
				// information to the map, other than to potentially shrink it.
				if (!writeType.sizeRange.rangeIncludesLong(
						newMap.mapSize.toLong()))
				{
					throw VariableSetException(
						E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
				}
			}
			else
			{
				// Do a full type-check.
				if (!newMap.isInstanceOf(writeType))
					throw VariableSetException(
						E_CANNOT_STORE_INCORRECTLY_TYPED_VALUE)
			}
			if (o_CompareAndSwapValuesNoCheck(self, oldValue, newMap))
			{
				break
			}
		}
	}

	@Throws(VariableGetException::class)
	override fun o_VariableMapHasKey(
		self: AvailObject, key: A_BasicObject): Boolean
	{
		val map = o_GetValue(self)
		assert(map.isMap)
		return map.hasKey(key)
	}

	override fun o_ClearValue(self: AvailObject)
	{
		handleVariableWriteTracing(self)
		self.setVolatileSlot(VALUE, nil)
		recordWriteToSharedVariable()
	}

	override fun o_AddWriteReactor(
		self: AvailObject,
		key: A_Atom,
		reactor: VariableAccessReactor)
	{
		recordReadFromSharedVariable(self)
		super.o_AddWriteReactor(self, key, reactor)
	}

	@Throws(AvailException::class)
	override fun o_RemoveWriteReactor(self: AvailObject, key: A_Atom)
	{
		recordReadFromSharedVariable(self)
		super.o_RemoveWriteReactor(self, key)
	}

	override fun o_WriteTo(self: AvailObject, writer: JSONWriter) =
		writer.writeObject {
			at("kind") { write("variable") }
			at("variable type") { self[KIND].writeTo(writer) }
			at("value") { self.value().writeSummaryTo(writer) }
		}

	override fun o_WriteSummaryTo(self: AvailObject, writer: JSONWriter) =
		writer.writeObject {
			at("kind") { write("variable") }
			at("variable type") { self.kind().writeSummaryTo(writer) }
		}

	/**
	 * Extract the given variable's write-reactors map, and pass it into the
	 * [body] function.  If toModify is true, initialize the field if
	 * needed.  If toModify is false and the field has not yet been set, use
	 * `null` instead.  Ensure that the map can not be read or written by
	 * other threads during the body.
	 *
	 * Use a lock on the write-reactors map itself, to ensure atomicity.  If the
	 * map is `null` (and therefore toModify was false), don't lock anything.
	 *
	 * @param T
	 *   The type of value produced by the body and returned by this method.
	 * @param self
	 *   The [A_Variable] to examine and/or update.
	 * @param toModify
	 *   Whether to initialize the field if it has not yet been initialized.
	 * @param body
	 *   A function that runs with either this variable's [MutableMap] from
	 *   [A_Atom] to [VariableDescriptor.VariableAccessReactor], or `null`.
	 */
	override fun<T> withWriteReactorsToModify(
		self: AvailObject,
		toModify: Boolean,
		body: (MutableMap<A_Atom, VariableAccessReactor>?)->T
	): T = super.withWriteReactorsToModify(self, toModify) { map ->
		if (map == null)
		{
			assert(!toModify)
			body(null)
		}
		else
		{
			synchronized(map) { body(map) }
		}
	}

	companion object
	{
		/**
		 * Indicate in the current fiber's
		 * [availLoader][Interpreter.availLoader] that a shared variable has
		 * just been modified.
		 */
		@JvmStatic
		protected fun recordWriteToSharedVariable()
		{
			AvailLoader.currentLoaderOrNull()?.statementCanBeSummarized(false)
		}

		/**
		 * Indicate in the current fiber's
		 * [availLoader][Interpreter.availLoader] that a shared variable has
		 * just been read.
		 *
		 * @param self
		 *   The shared variable that was read.
		 */
		private fun recordReadFromSharedVariable(self: AvailObject)
		{
			val loader = AvailLoader.currentLoaderOrNull() ?: return
			if (loader.statementCanBeSummarized()
				&& self.volatileSlot(VALUE).notNil
				&& !self.valueWasStablyComputed())
			{
				loader.statementCanBeSummarized(false)
			}
		}

		/**
		 * Create a [shared][Mutability.SHARED] [variable][A_Variable] that has
		 * the given characteristics.
		 *
		 * @param kind
		 *   The [variable&#32;type][VariableTypeDescriptor].
		 * @param hash
		 *   The hash of the variable.
		 * @param value
		 *   The contents of the variable.
		 * @param writeReactors
		 *   The write reactors weak map pojo to use.
		 * @return
		 *   The shared variable.
		 */
		fun createSharedLike(
			kind: A_Type,
			hash: Int,
			value: A_BasicObject,
			writeReactors: A_BasicObject
		): AvailObject
		{
			assert(kind.descriptor().isShared)
			assert(value.descriptor().isShared)
			return mutableInitial.create {
				setSlot(KIND, kind)
				setSlot(HASH_ALWAYS_SET, hash)
				setSlot(VALUE, value)
				setSlot(WRITE_REACTORS, writeReactors)
				setDescriptor(shared)
			}
		}

		/**
		 * The mutable [VariableSharedDescriptor]. Exists only to support
		 * creation.
		 */
		private val mutableInitial = VariableSharedDescriptor(
			Mutability.MUTABLE,
			TypeTag.VARIABLE_TAG,
			ObjectSlots::class.java,
			IntegerSlots::class.java)

		/** The shared [VariableSharedDescriptor]. */
		val shared = VariableSharedDescriptor(
			Mutability.SHARED,
			TypeTag.VARIABLE_TAG,
			ObjectSlots::class.java,
			IntegerSlots::class.java)
	}
}
