// Copyright (c) 2020-2022 Tesla (Yinsen) Zhang.
// Use of this source code is governed by the MIT license that can be found in the LICENSE.md file.
package org.aya.concrete.error;

import kala.collection.immutable.ImmutableSeq;
import org.aya.distill.BaseDistiller;
import org.aya.pretty.doc.Doc;
import org.aya.pretty.doc.Style;
import org.aya.util.binop.Assoc;
import org.aya.util.binop.BinOpSet;
import org.aya.util.distill.DistillerOptions;
import org.aya.util.error.SourcePos;
import org.aya.util.reporter.Problem;
import org.jetbrains.annotations.NotNull;

import java.util.Comparator;

public interface OperatorError extends Problem {
  @Override default @NotNull Problem.Severity level() {return Problem.Severity.ERROR;}
  @Override default @NotNull Stage stage() {return Stage.PARSE;}

  record BadBindBlock(@NotNull SourcePos sourcePos, @NotNull String op) implements OperatorError {
    @Override public @NotNull Doc describe(@NotNull DistillerOptions options) {
      return Doc.sep(Doc.plain(op), Doc.english("is not an operator. It cannot have bind block."));
    }
  }

  record Fixity(
    @NotNull String op1, @NotNull Assoc assoc1,
    @NotNull String op2, @NotNull Assoc assoc2,
    @Override @NotNull SourcePos sourcePos
  ) implements OperatorError {
    @Override
    public @NotNull Doc describe(@NotNull DistillerOptions options) {
      return Doc.sep(
        Doc.english("Cannot figure out computation order because"),
        Doc.styled(Style.code(), Doc.plain(op1)),
        Doc.parened(Doc.plain(assoc1.name())),
        Doc.plain("and"),
        Doc.styled(Style.code(), Doc.plain(op2)),
        Doc.parened(Doc.plain(assoc1.name())),
        reason()
      );
    }

    private @NotNull Doc reason() {
      return assoc1 == assoc2 && assoc1 == Assoc.Infix
        ? Doc.english("share the same precedence but no associativity was specified.")
        : Doc.english("share the same precedence but don't share the same associativity.");
    }

    @Override public @NotNull Doc hint(@NotNull DistillerOptions options) {
      return Doc.english("Make them both left/right-associative to resolve this problem.");
    }
  }

  record Precedence(
    @NotNull String op1,
    @NotNull String op2,
    @Override @NotNull SourcePos sourcePos
  ) implements OperatorError {
    @Override public @NotNull Doc describe(@NotNull DistillerOptions options) {
      return Doc.sep(
        Doc.english("Ambiguous operator precedence detected between"),
        Doc.styled(Style.code(), Doc.plain(op1)),
        Doc.plain("and"),
        Doc.styled(Style.code(), Doc.plain(op2))
      );
    }

    @Override public @NotNull Doc hint(@NotNull DistillerOptions options) {
      return Doc.sep(Doc.plain("Use"),
        Doc.styled(BaseDistiller.KEYWORD.and().code(), Doc.plain("tighter/looser")),
        Doc.english("clause or insert parentheses to make it clear."));
    }
  }

  record SelfBind(@Override @NotNull SourcePos sourcePos) implements OperatorError {
    @Override public @NotNull Doc describe(@NotNull DistillerOptions options) {
      return Doc.english("Self bind is not allowed");
    }
  }

  record Circular(
    @NotNull ImmutableSeq<BinOpSet.BinOP> items
  ) implements OperatorError {
    @Override public @NotNull SourcePos sourcePos() {
      return items.view().map(BinOpSet.BinOP::firstBind)
        .max(Comparator.comparingInt(SourcePos::endLine));
    }

    @Override public @NotNull Doc describe(@NotNull DistillerOptions options) {
      return Doc.sep(
        Doc.english("Circular precedence found between"),
        Doc.commaList(items.view().map(BinOpSet.BinOP::name).toImmutableSeq()
          .sorted().view().map(Doc::plain))
      );
    }
  }
}
