#ifndef UAST_H_
#define UAST_H_

#include <stdbool.h>
#include <stdint.h>
#include <stddef.h>
#include <stdlib.h>

#define UAST_HASH_SIZE 32

// NodeHandle is an opaque node handle that client should use to track nodes passed to libuast.
// A handle can either be a real pointer to the node, or an ID value that client assigns to the node.
typedef uintptr_t NodeHandle;

// UastHandle is an opaque UAST context handle allocated by the client. It can be used to attach additional
// information to the UAST context. Implementation may decide to ignore the UAST context handle and interpret
// NodeHandle as pointers to node objects.
typedef uintptr_t UastHandle;

typedef enum {
    NODE_NULL,
    NODE_OBJECT,
    NODE_ARRAY,
    NODE_STRING,
    NODE_INT,
    NODE_UINT,
    NODE_FLOAT,
    NODE_BOOL,
} NodeKind;

typedef struct Uast Uast;

// NodeIface is an interface for UAST nodes that client should implement to bind to libuast.
// Each function of the interface receives a UastHandle that can be used by the client to store
// all handle-to-node mappings for this particular UAST context. See UastHandle and NodeHandle for more details.
typedef struct NodeIface {
    NodeKind (*Kind)(const Uast*, NodeHandle);

    char *   (*AsString)(const Uast*, NodeHandle);
    int64_t  (*AsInt)   (const Uast*, NodeHandle);
    uint64_t (*AsUint)  (const Uast*, NodeHandle);
    double   (*AsFloat) (const Uast*, NodeHandle);
    bool     (*AsBool)  (const Uast*, NodeHandle);

    size_t (*Size)(const Uast*, NodeHandle);

    char *     (*KeyAt)  (const Uast*, NodeHandle, size_t);
    NodeHandle (*ValueAt)(const Uast*, NodeHandle, size_t);


    NodeHandle (*NewObject)(const Uast*, size_t size);
    NodeHandle (*NewArray) (const Uast*, size_t size);
    NodeHandle (*NewString)(const Uast*, const char * str);
    NodeHandle (*NewInt)   (const Uast*, int64_t);
    NodeHandle (*NewUint)  (const Uast*, uint64_t);
    NodeHandle (*NewFloat) (const Uast*, double);
    NodeHandle (*NewBool)  (const Uast*, bool);

    void (*SetValue)(const Uast*, NodeHandle, size_t, NodeHandle);
    void (*SetKeyValue)(const Uast*, NodeHandle, const char *, NodeHandle);

} NodeIface;

typedef enum {
    // ANY_ORDER is a native iteration order of the tree. It's the fastest iteration order that lists all nodes in the tree.
    // The iteration order is not guaranteed to be the same for consecutive iterations over the same tree.
    // This order is more suitable for searching for nodes in the fastest way possible.
    ANY_ORDER,
    // PRE_ORDER is a pre-order depth-first search.
    PRE_ORDER,
    // POST_ORDER is a post-order depth-first search.
    POST_ORDER,
    // LEVEL_ORDER is a breadth-first search.
    LEVEL_ORDER,
    // CHILDREN_ORDER is similar to LEVEL_ORDER, but list only the first level.
    CHILDREN_ORDER,
    // POSITION_ORDER enumerates nodes all nodes sorted by the position in the source file.
    POSITION_ORDER,
} TreeOrder;

#define UAST_CALL(u, name, ...) u->iface->name(u, __VA_ARGS__)

// Uast stores the general context required for library functions.
// It must be initialized with UastNew passing a valid implementation of the NodeIface interface.
// Once it is not used anymore, it shall be released calling `UastFree`.
typedef struct Uast {
 // iface is an implementation of the node interface that will be used for this UAST context.
 NodeIface  *iface;
 // handle is an internal UAST context handle defined by the libuast. It shouldn't be changed our used by the client.
 uintptr_t  handle;
 // ctx is an optional UAST handle that libuast will pass to each node interface function.
 // It can be used to track different UAST contexts in the client code.
 UastHandle ctx;
 // root is an optional root node handle that will be used by default if Filter, Encode and other operations.
 NodeHandle root;
} Uast;

// An UastIterator is used to keep the state of the current iteration over the tree.
// It's initialized with UastIteratorNew, used with UastIteratorNext and freed
// with UastIteratorFree.
typedef struct UastIterator {
  const Uast *ctx;
  uintptr_t handle;
} UastIterator;

typedef enum { UAST_BINARY = 0, UAST_YAML = 1 } UastFormat;

typedef enum {
    HASH_NO_POS = 0x1,
} HashFlags;

// UastLoad copies the node from a source context into the destination.
//
// Since contexts might be backed by different node interface implementations,
// this functions allows to load UAST to and from the libuast-owned memory.
static NodeHandle UastLoad(const Uast *src, NodeHandle n, const Uast *dst) {
    NodeKind kind = UAST_CALL(src, Kind, n);

    if (kind == NODE_NULL) {
        return 0;
    } else if (kind == NODE_OBJECT) {
      size_t sz = UAST_CALL(src, Size, n);

      NodeHandle m = UAST_CALL(dst, NewObject, sz);
      size_t i;

      for (i = 0; i < sz; i++) {
        char* k = UAST_CALL(src, KeyAt, n, i);

        if (!k) {
          return 0;
        }

        NodeHandle v = UAST_CALL(src, ValueAt, n, i);
        v = UastLoad(src, v, dst);
        UAST_CALL(dst, SetKeyValue, m, k, v);
        free(k);
      }
      return m;
    } else if (kind == NODE_ARRAY) {
      size_t sz = UAST_CALL(src, Size, n);

      NodeHandle m = UAST_CALL(dst, NewArray, sz);
      size_t i;
      for (i = 0; i < sz; i++) {
        NodeHandle v = UAST_CALL(src, ValueAt, n, i);
        v = UastLoad(src, v, dst);
        UAST_CALL(dst, SetValue, m, i, v);
      }
      return m;
    } else if (kind == NODE_STRING) {
      char *s = UAST_CALL(src, AsString, n);
      NodeHandle m = UAST_CALL(dst, NewString, s);
      free(s);
      return m;
    } else if (kind == NODE_INT) {
      return UAST_CALL(dst, NewInt, UAST_CALL(src, AsInt, n));
    } else if (kind == NODE_UINT) {
      return UAST_CALL(dst, NewUint, UAST_CALL(src, AsUint, n));
    } else if (kind == NODE_FLOAT) {
      return UAST_CALL(dst, NewFloat, UAST_CALL(src, AsFloat, n));
    } else if (kind == NODE_BOOL) {
      return UAST_CALL(dst, NewBool, UAST_CALL(src, AsBool, n));
    }

    return 0;
}

// UastMemStats holds memory statistics for libuast.
typedef struct UastMemStats {
 uint64_t allocated; // bytes allocated for live objects
 uint64_t objects;   // number of live objects
} UastMemStats;

// UastSourceIndex represents a positional index for a source file.
// It's initialized with UastSourceIndexNew and freed with UastSourceIndexFree.
typedef struct UastSourceIndex {
  uintptr_t handle;
} UastSourceIndex;

// UastLineCol is a one-based line-column position in a source file.
typedef struct UastLineCol {
  int line;
  int col;
} UastLineCol;


#ifdef __cplusplus
extern "C" {
#endif

// UastNew initializes a new UAST context that will use a provided node interface as an implementation.
// This allows libuast to work with every language's native node data structures. Client can pass
// an additional UastHandle to distinguish between different UAST context instances.
//
// The returned context pointer is guaranteed to be not NULL. Client should check LastError before
// using the context and deallocate it with UastFree in case of an error occurs, or when the context
// is no longer needed.
extern Uast* UastNew(NodeIface* iface, UastHandle ctx);

// UastDecode accepts a pointer to a buffer with a specified size and decodes the content into
// a new UAST structure.
//
// The new UAST context will use internal node interface implementation and all the nodes will
// be managed by libuast.
//
// The returned context pointer is guaranteed to be not NULL. Client should check LastError before
// using the context and deallocate it with UastFree in case of an error occurs, or when the context
// is no longer needed.
extern Uast* UastDecode(void* p, size_t sz, UastFormat format);

// UastEncode encodes a given node and all it's children to a specified encoding. It returns a pointer to the bytes buffer
// and writes the size of the buffer to the size pointer.
extern void* UastEncode(Uast* ctx, NodeHandle node, size_t* size, UastFormat format);

// UastFree releases resources associated with this UAST context.
extern void UastFree(Uast* ctx);

// UastLastError return the last encountered error in this context, if any.
extern char* UastLastError(Uast* ctx);

// UastFilter filters UAST starting from a given node with XPath query and returns an iterator for results.
// Caller should free an iterator by calling UastIteratorFree.
extern UastIterator* UastFilter(Uast* ctx, NodeHandle node, char* query);

// UastSetError sets an error state for this context. It can be used inside interface functions to indicate an error to the caller.
extern void UastSetError(Uast* ctx, char* str);

// UastIteratorNew starts an iterator from a given node in a specified order. Caller should free an iterator by calling UastIteratorFree.
extern UastIterator* UastIteratorNew(Uast* ctx, NodeHandle node, TreeOrder order);

// UastIteratorFree release all resources associated with an iterator.
extern void UastIteratorFree(UastIterator* it);

// UastIteratorNext return the next node on this iterator or 0 if iterator is empty.
extern NodeHandle UastIteratorNext(UastIterator* it);

// UastEqual checks if nodes are equal.
extern bool UastEqual(Uast* ctx1, NodeHandle node1, Uast* ctx2, NodeHandle node2);

// UastHash computes s hash of a given node and stores it to dst.
// Allocated buffer should be at least UAST_HASH_SIZE long.
extern void UastHash(Uast* ctx, NodeHandle node, void* dst, HashFlags flags);

// RoleNameForId converts a role id to a string role name.
extern char* RoleNameForId(int id);

// RoleIdForName converts a role name to an id and returns 0 if no such role was defined.
extern int RoleIdForName(char* name);

// UastReadMemStats fills in current memory statistics. This call may affect libuast performance.
extern void UastReadMemStats(UastMemStats* st);

// UastSourceIndexNew creates a positional index for a source file. An index can be used
// to convert byte-based node offsets to Unicode character offsets.
extern UastSourceIndex* UastSourceIndexNew(void* source, size_t size);

// UastSourceIndexFree release all resources associated with an index.
extern void UastSourceIndexFree(UastSourceIndex* idx);

// UastSourceIndex_LastError returns the last error returned by index operations.
extern char* UastSourceIndex_LastError(UastSourceIndex* idx);

// UastSourceIndex_ClearError clears an error state.
extern void UastSourceIndex_ClearError(UastSourceIndex* idx);

// UastSourceIndex_FromLineCol converts one-based line-column pair (in bytes) in the indexed
// source file to a zero-based byte offset. It return -1 in case of failure.
extern int UastSourceIndex_FromLineCol(UastSourceIndex* idx, int line, int col);

// UastSourceIndex_ToLineCol converts zero-based byte offset in the indexed source
// file to a one-based line and one-based column pair (in bytes).
// It return a UastLineCol with both elements set to -1 in case of failure.
extern UastLineCol UastSourceIndex_ToLineCol(UastSourceIndex* idx, int off);

// UastSourceIndex_FromUnicode converts zero-based Unicode character offset in the indexed
// source file to a zero-based byte offset. It return -1 in case of failure.
extern int UastSourceIndex_FromUnicode(UastSourceIndex* idx, int off);

// UastSourceIndex_ToUnicode converts zero-based byte offset in the indexed source file to
// a zero-based Unicode character offset. It return -1 in case of failure.
extern int UastSourceIndex_ToUnicode(UastSourceIndex* idx, int off);

// UastSourceIndex_FromUTF16 converts zero-based UTF-16 code point offset in the indexed
// source file to a zero-based byte offset. It return -1 in case of failure.
extern int UastSourceIndex_FromUTF16(UastSourceIndex* idx, int off);

// UastSourceIndex_ToUTF16 converts zero-based byte offset in the indexed source file to
// a zero-based UTF-16 code point offset. It return -1 in case of failure.
extern int UastSourceIndex_ToUTF16(UastSourceIndex* idx, int off);

// UastSourceIndex_ToUnicodeLineCol converts zero-based byte offset in the indexed source
// file to a one-based line and one-based column pair (in Unicode characters).
// It return a UastLineCol with both elements set to -1 in case of failure.
extern UastLineCol UastSourceIndex_ToUnicodeLineCol(UastSourceIndex* idx, int off);

// UastSourceIndex_ToUTF16LineCol converts zero-based byte offset in the indexed source
// file to a one-based line and one-based column pair (in UTF-16 code units).
// It return a UastLineCol with both elements set to -1 in case of failure.
extern UastLineCol UastSourceIndex_ToUTF16LineCol(UastSourceIndex* idx, int off);

#ifdef __cplusplus
}
#endif


#endif // UAST_H_
