package org.bdware.bdosclient;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.bdosclient.bean.PkgInfo;
import org.bdware.bdosclient.bean.SearchResult;
import org.bdware.doip.audit.client.AuditDoipClient;
import org.bdware.doip.audit.writer.NoneAuditConfig;
import org.bdware.doip.codec.digitalObject.DigitalObject;
import org.bdware.doip.codec.digitalObject.DoType;
import org.bdware.doip.codec.doipMessage.DoipMessage;
import org.bdware.doip.codec.doipMessage.DoipMessageFactory;
import org.bdware.doip.codec.doipMessage.DoipResponseCode;
import org.bdware.doip.codec.operations.BasicOperations;
import org.bdware.doip.encrypt.SM2Signer;
import org.bdware.doip.endpoint.client.ClientConfig;
import org.bdware.doip.endpoint.client.DoipMessageCallback;
import org.bdware.irp.client.IrpClient;
import org.bdware.irp.stateinfo.StateInfoBase;
import org.zz.gmhelper.SM2KeyPair;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class CodeRepoClient {
    private final SM2KeyPair keyPair;
    private final String repoDoid;
    private final IrpClient irpClient;
    AuditDoipClient client;
    static Logger LOGGER = LogManager.getLogger(CodeRepoClient.class);

    static ExecutorService threadPool = Executors.newFixedThreadPool(5);

    public static String getDefaultCodeRepo(String prefix) {
        return prefix.replaceAll("/.*$", "") + "/CodeRepository";
    }

    public CodeRepoClient(String repoDoid, IrpClient irpClient, SM2KeyPair keyPair) {
        this.keyPair = keyPair;
        this.repoDoid = repoDoid;
        this.irpClient = irpClient;
    }

    private void initClient() {
        try {
            if (client != null && client.isConnected()) {
                client.reconnect();
                return;
            }
            StateInfoBase resolveResult = irpClient.resolve(repoDoid);
            NoneAuditConfig config = new NoneAuditConfig(null);
            client = new AuditDoipClient(config, new SM2Signer(keyPair));
            client.connect(ClientConfig.fromUrl(resolveResult.handleValues.get("address").getAsString()));
            client.setTimeout(30000);
        } catch (Exception e) {
            e.printStackTrace();
            client = null;
        }
    }

    public void createAndUpload(String path, ProgressCallback progressCallback) {
        initClient();
        createYPK(path, new DoipMessageCallback() {
            @Override
            public void onResult(DoipMessage msg) {
                //if success
                progressCallback.onStart(msg.header.parameters.id);
                if (msg.header.parameters.response != DoipResponseCode.Success) {
                    progressCallback.onError("error in create doid", msg);
                    return;
                }
                threadPool.execute(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            int chunk = 1000;
                            if (msg.header.parameters.attributes.has("chunkSize")) {
                                chunk = msg.header.parameters.attributes.get("chunkSize").getAsInt();
                            }
                            uploadChuckByChunk(msg.header.parameters.id, path, chunk, progressCallback);
                        } catch (Exception e) {
                            ByteArrayOutputStream bo = new ByteArrayOutputStream();
                            e.printStackTrace(new PrintStream(bo));
                            progressCallback.onError("error in upload:" + new String(bo.toByteArray()), null);
                            e.printStackTrace();
                        }
                    }
                });
            }
        });
        return;
    }

    private void uploadChuckByChunk(String doid, String path, long chunkSize, ProgressCallback progressCallback) throws Exception {
        long total = new File(path).length();
        int step = 5;
        int progress = step;
        RandomAccessFile accessFile = new RandomAccessFile(new File(path), "r");
        byte[] buff = new byte[(int) chunkSize];
        for (int i = 0; i < total; i += chunkSize) {
            long len = accessFile.read(buff);
            DoipMessageFactory.DoipMessageBuilder builder = (new DoipMessageFactory.DoipMessageBuilder()).createRequest(doid, BasicOperations.Update.getName());
            if (len == chunkSize)
                builder.setBody(buff);
            else {
                byte[] readed = new byte[(int) len];
                System.arraycopy(buff, 0, readed, 0, (int) len);
                builder.setBody(readed);
            }
            builder.addAttributes("currentPos", i);
            if (i + chunkSize >= total) {
                builder.addAttributes("lastChunk", "true");
            }
            DoipMessage resp = client.sendMessageSync(builder.create());
            if (resp.header.parameters.response != DoipResponseCode.Success) {
                progressCallback.onError("unknowns error in uploading", resp);
                return;
            }
            if (i * 100 / total >= progress) {
                progress += step;
                progressCallback.onProgress(doid, i, (int) total);
            }
        }
        progressCallback.onFinish(doid);
    }

    //path= /a/b/c/ypkname-ypkversion.ypk
    private void createYPK(String path, DoipMessageCallback cb) {
        DigitalObject digitalObject = new DigitalObject(null, DoType.DO);
        digitalObject.id = "/";
        digitalObject.attributes = new JsonObject();
        digitalObject.attributes.addProperty("owner", keyPair.getPublicKeyStr());
        String fileName = new File(path).getName();
        int offset = fileName.lastIndexOf("-");
        String ypkName = fileName.substring(0, offset);
        int suffixOffset = fileName.lastIndexOf(".ypk");
        String version = fileName.substring(offset + 1, suffixOffset);
        digitalObject.attributes.addProperty("ypkName", ypkName);
        digitalObject.attributes.addProperty("version", version);
        digitalObject.attributes.addProperty("md5", calFileMd5(path));
        digitalObject.attributes.addProperty("fileSize", new File(path).length());
        client.create(repoDoid, digitalObject, cb);
        return;
    }

    public void deleteYPK(String doid, DoipMessageCallback cb) {
        initClient();
        client.delete(doid, cb);
    }

    public void retrieveBCO(String doid, DoipMessageCallback cb) {
        initClient();
        client.retrieve(doid, "-1", false, cb);
    }

    public void listPackages(int offset, int count, DoipMessageCallback cb) {
        initClient();
        DoipMessage msg = (new DoipMessageFactory.DoipMessageBuilder()).createRequest(repoDoid, BasicOperations.Retrieve.getName()).create();
        msg.header.parameters.addAttribute("element", "0");
        msg.header.parameters.addAttribute("includeElementData", "true");
        msg.header.parameters.addAttribute("offset", offset);
        msg.header.parameters.addAttribute("count", count);
        client.sendMessage(msg, cb);
    }

    public void downloadYPK(String doid, ProgressCallback progressCallback, File output) {
        initClient();
        DoipMessage message = client.retrieveSync(doid, "", false);
        if (message.header.parameters.response == DoipResponseCode.Success) {
            JsonObject resolved = JsonParser.parseString(message.body.getDataAsJsonString()).getAsJsonObject();
            resolved.addProperty("chunkSize", message.header.parameters.attributes.get("chunkSize").getAsString());
            progressCallback.onStart(doid);
            threadPool.execute(new Runnable() {
                @Override
                public void run() {
                    downloadInternal(doid, resolved, progressCallback, output);
                }
            });
        }
    }

    private void downloadInternal(String doid, JsonObject resolved, ProgressCallback progressCallback, File output) {
        int total = resolved.get("fileSize").getAsInt();
        int chunkSize = resolved.get("chunkSize").getAsInt();
        try {
            File parent = output.getParentFile();
            if (!parent.exists()) parent.mkdirs();
            if (output.exists()) output.delete();
            output.createNewFile();
            FileOutputStream fout = new FileOutputStream(output);
            double step = 0.05;
            double progress = 0;
            for (int i = 0; i * chunkSize < total; i++) {
                DoipMessage doipMessage = client.retrieveSync(doid, i + "", true);
                fout.write(doipMessage.body.getEncodedData());
                if ((i * chunkSize + 0.0D) / (total + 0.0D) > progress) {
                    progressCallback.onProgress(doid, i * chunkSize, total);
                    progress += step;
                }
            }
            fout.close();
            String md5 = calFileMd5(output.getAbsolutePath());
            if (!md5.equals(resolved.get("md5").getAsString())) {
                progressCallback.onError("verify failed! redownload and check", null);
            }
            progressCallback.onFinish(doid);
        } catch (Exception e) {
            ByteArrayOutputStream bo = new ByteArrayOutputStream();
            e.printStackTrace(new PrintStream(bo));
            progressCallback.onError(new String(bo.toByteArray()), null);
        }
    }

    public String updatePkg(PkgInfo info) {
        initClient();
        String doid = repoDoid + "/" + info.packageName;
        DoipMessageFactory.DoipMessageBuilder builder = new DoipMessageFactory.DoipMessageBuilder();
        builder.createRequest(doid, BasicOperations.Update.getName());
        builder.setBody(new Gson().toJson(info).getBytes(StandardCharsets.UTF_8));
        try {
            DoipMessage result = client.sendMessageSync(builder.create());
            return result.body.getDataAsJsonString();
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public interface ProgressCallback {
        public void onStart(String doId);

        public void onProgress(String doId, int currentChunk, int totalChunk);

        public void onFinish(String doId);

        void onError(String message, DoipMessage msg);
    }

    public String calFileMd5(String path) {
        String md5 = "errormd5";
        try {
            md5 = DigestUtils.md5Hex(new FileInputStream(path));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return md5;
    }

    enum SearchType {
        Description("description"), PackageName("packageName");
        private final String type;

        SearchType(String type) {
            this.type = type;
        }

        public String getType() {
            return type;
        }

    }

    public SearchResult search(SearchType type, String queryStr, int offset, int count) {
        initClient();
        JsonObject arguments = new JsonObject();
        arguments.addProperty("type", type.getType());
        arguments.addProperty("queryStr", queryStr);
        arguments.addProperty("offset", offset);
        arguments.addProperty("count", count);
        DoipMessageFactory.DoipMessageBuilder builder = new DoipMessageFactory.DoipMessageBuilder();
        builder.createRequest("empty", BasicOperations.Search.getName());
        DoipMessage msg = builder.create();
        msg.header.parameters.attributes = arguments;
        DoipMessage result = client.sendMessageSync(msg);
        try {
            SearchResult result2 = new Gson().fromJson(result.body.getDataAsJsonString(), SearchResult.class);
            return result2;
        } catch (Exception e) {
            e.printStackTrace();
            SearchResult empty = new SearchResult();
            empty.setTotal(-1);
            return empty;
        }
    }
}
