package org.bdware.bdosclient;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.doip.audit.client.AuditDoipClient;
import org.bdware.doip.audit.writer.NoneAuditConfig;
import org.bdware.doip.codec.digitalObject.DigitalObject;
import org.bdware.doip.codec.digitalObject.DoType;
import org.bdware.doip.codec.doipMessage.DoipMessage;
import org.bdware.doip.codec.doipMessage.DoipMessageFactory;
import org.bdware.doip.codec.exception.DoipConnectException;
import org.bdware.doip.codec.operations.BasicOperations;
import org.bdware.doip.encrypt.SM2Signer;
import org.bdware.doip.endpoint.client.ClientConfig;
import org.bdware.doip.endpoint.client.DoipMessageCallback;
import org.bdware.irp.client.IrpClient;
import org.bdware.irp.stateinfo.StateInfoBase;
import org.zz.gmhelper.SM2KeyPair;

public class BDRepoClient {
    static Logger LOGGER = LogManager.getLogger(BDRepoClient.class);
    private final SM2KeyPair keyPair;
    private final IrpClient irpClient;

    private final String repoDoid;
    AuditDoipClient client;

    public BDRepoClient(String repoDoid, IrpClient irpClient, SM2KeyPair keyPair) {
        this.keyPair = keyPair;
        this.irpClient = irpClient;
        this.repoDoid = repoDoid;
        try {
            StateInfoBase resolveResult = irpClient.resolve(repoDoid);
            NoneAuditConfig config = new NoneAuditConfig(null);
            client = new AuditDoipClient(config, new SM2Signer(keyPair, true, false));
            client.connect(ClientConfig.fromUrl(resolveResult.handleValues.get("address").getAsString()));
            client.setTimeout(30000);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public static class StartBDOResult {
        public int shortId;
        public String publicKey;
        public JsonElement onCreate;
        public JsonElement doipOnStart;
        public int doipListenerPort;
        public boolean isValid;
        public transient DoipMessage originalMessage;
        public long executionTime;
    }

    public interface StartBDOResultCallback {
        public void onResult(StartBDOResult result);
    }


    //{"action":"startContractByYPK","createParam":123,"isDebug":false,"dumpPeriod":"","isPrivate":false,"owner":"04d1924329f72ced148f6f333fb985ccbaa31b1e3aacf10be5f43d4a4ff5ad88899a005e79e37fc06993e1d66ada8cf8b711cb36f59538bb7d3e39e70fa9360ddd","requestID":"1679227007551","path":"/NanningAppC_2022-11-08-12_26_39_Auto.ypk","signature":"304502204123278e7d8932661803a257c3bd41a41ef1d94a0fdbcc3014d52d185728449c0221008a11a75a6eea19808095b6d2bdd744950ec9d4f96661cb781c97c015f2518c65","script":"empty"}
    public void createBDO(String bcoId, int shardingId, JsonElement createParam, int doipStartPort, SM2KeyPair keyPair, StartBDOResultCallback cb) {
        ensureConnected();
        StartBDOResult result = new StartBDOResult();
        result.executionTime = System.currentTimeMillis();
        if (doipStartPort > 0)
            createParam.getAsJsonObject().addProperty("doipStartPort", doipStartPort);
        DoipMessageCallback converter = new DoipMessageCallback() {
            @Override
            public void onResult(DoipMessage msg) {
                try {
                    JsonObject dataObj = JsonParser.parseString(msg.body.getDataAsJsonString()).getAsJsonObject();
                    result.originalMessage = msg;
                    result.doipListenerPort = dataObj.get("doipListenPort").getAsInt();
                    result.doipOnStart = dataObj.get("doipModuleStartResult");
                    result.onCreate = dataObj.get("onCreate").getAsJsonObject();
                    result.publicKey = dataObj.get("loadContract").getAsJsonObject().get("result").getAsString();
                    result.isValid = true;
                    result.shortId = result.publicKey.hashCode();
                } catch (Exception e) {
                    result.isValid = false;
                    result.originalMessage = msg;
                }
                result.executionTime = System.currentTimeMillis() - result.executionTime;
                cb.onResult(result);
            }
        };
        JsonObject jo = new JsonObject();
        jo.add("createParam", createParam);
        jo.addProperty("bcoId", bcoId);
        // !IMPORTANT contractExecType: Sole Sharding
        //     jo.addProperty("contractExecType","Sole");
        jo.addProperty("shardingId", shardingId);
        if (keyPair != null)
            jo.add("sm2KeyPair", JsonParser.parseString(keyPair.toJson()));
        DigitalObject digitalObject = new DigitalObject("unknown", DoType.DO);
        digitalObject.attributes = jo;
        ensureConnected();
        client.create(repoDoid, digitalObject, converter);
        return;
    }

    //{"action":"killContractProcess","requestID":"1679228340097","id":"-1859639959"}
    public void deleteBDO(String idOrName, DoipMessageCallback cb) {
        ensureConnected();
        DoipMessageFactory.DoipMessageBuilder builder = new DoipMessageFactory.DoipMessageBuilder();
        builder.createRequest(repoDoid, BasicOperations.Delete.getName());
        builder.addAttributes("elementId", idOrName);
        client.sendRawMessage(builder.create(), cb);
    }

    //idOrName为空时，为retrieveList
    //由于每个节点的不太多，所以是返回全部正在运行的BDO?

    public void retrieveBDO(String idOrName, DoipMessageCallback cb) {
        ensureConnected();
        client.retrieve(repoDoid, idOrName, false, cb);
    }

    public void ensureConnected() {
        if (!client.isConnected()) {
            try {
                client.reconnect();
            } catch (DoipConnectException e) {
                throw new RuntimeException(e);
            }
        }
    }

}
