/*
 *    Copyright (c) [2021] [Peking University]
 *    [BDWare DOIP SDK] is licensed under Mulan PSL v2.
 *    You can use this software according to the terms and conditions of the Mulan PSL v2.
 *    You may obtain a copy of Mulan PSL v2 at:
 *             http://license.coscl.org.cn/MulanPSL2
 *    THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 *    See the Mulan PSL v2 for more details.
 */

package org.bdware.doip.endpoint.client;

import io.netty.util.HashedWheelTimer;
import io.netty.util.Timeout;
import io.netty.util.TimerTask;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.doip.codec.doipMessage.DoipMessage;
import org.bdware.doip.codec.doipMessage.DoipMessageFactory;

import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;


public class ResponseWait {
    static Logger LOGGER = LogManager.getLogger(ResponseWait.class);
    public static final HashedWheelTimer HASHED_WHEEL_TIMER = new HashedWheelTimer(Executors.defaultThreadFactory(), 5, TimeUnit.MILLISECONDS, 2);
    // use static map to ensure requestid is !UNIC!
    // in a client(With multiple connection)
    final static Map<Integer, DoipMessageCallback> waitObj = new ConcurrentSkipListMap<>();


    public void wakeUpAndRemove(int requestID, DoipMessage result) {
        DoipMessageCallback ob = getAndRemove(requestID);
        wakeup(requestID, ob, result);
    }

    //DO NOT use synchronized like "private synchronized ..."
    //Because the waitObj is static
    private DoipMessageCallback getAndRemove(int requestID) {
        synchronized (waitObj) {
            DoipMessageCallback ob = waitObj.get(requestID);
            waitObj.remove(requestID);
            return ob;
        }
    }

    public boolean waitResponse(final int requestID, DoipMessageCallback cb) {
        return waitResponse(requestID, cb, 5);
    }

    public boolean waitResponse(int requestID, DoipMessageCallback cb, int seconds) {
        if (waitObj.size() > 1000) {
            return false;
        }
        synchronized (waitObj) {
            if (!waitObj.containsKey(requestID)) {
                waitObj.put(requestID, cb);
                TimerTask tt = new TimerTask() {
                    @Override
                    public void run(Timeout timeout) throws Exception {
                        DoipMessage timeOutMessage = DoipMessageFactory.createTimeoutResponse(requestID);
                        wakeUpAndRemove(requestID, timeOutMessage);
                    }
                };
                HASHED_WHEEL_TIMER.newTimeout(tt, seconds, TimeUnit.SECONDS);
                return true;
            }
            LOGGER.info("Return false, try again!");
            return false;
        }
    }


    //TODO just support for stream??
    // schedule timeout cleaner?
    public void wakeup(int requestID, DoipMessage result) {
        DoipMessageCallback ob = waitObj.get(requestID);
        wakeup(requestID, ob, result);
    }

    public void wakeup(int requestID, DoipMessageCallback ob, DoipMessage result) {
        if (ob == null) {
            //   LOGGER.info("cancel the timeout task!!" + requestID + ", please Check server status??");
            // @TODO actually we can stop the timeout runner.
        }
        if (ob != null) ob.onResult(result);
    }
}
