package org.bdware.doip.codec.v3;

import io.netty.buffer.ByteBuf;
import org.bdware.doip.codec.v3.headers.ExtensionPart;

import java.util.ArrayList;
import java.util.List;

public class DOIPV3Message {
    public byte majorVersion;
    public byte minorVersion;
    public int nextHeader;
    public long requestID;
    public short reservedFlags;

    public int reserved1;

    public int contentLength;
    public static int HeaderLength = 1 + 1 + 4 + 8 + 2 + 4 + 4;
    public transient List<ExtensionPart> extensionParts;

    public DOIPV3Message() {
        majorVersion = 3;
        minorVersion = 0;
    }

    public void writeTo(ByteBuf bf) {
        fixNextHeaderAndCalculateContentLength();
        bf.writeByte(majorVersion);
        bf.writeByte(minorVersion);
        bf.writeInt(nextHeader);
        bf.writeLong(requestID);
        bf.writeShort(reservedFlags);
        bf.writeInt(reserved1);
        bf.writeInt(contentLength);
        if (extensionParts != null) {
            for (ExtensionPart part : extensionParts)
                part.writeTo(bf);
        }

    }

    public void fixNextHeaderAndCalculateContentLength() {
        contentLength = 0;
        if (extensionParts.size() > 0) {
            ExtensionPart previous = null;
            for (ExtensionPart part : extensionParts) {
                contentLength += part.length();
                if (previous == null) nextHeader = part.getType();
                else previous.setNextType(part.getType());
                previous = part;
            }

        }

    }

    public void readFrom(ByteBuf bf) {
        majorVersion = bf.readByte();
        minorVersion = bf.readByte();
        nextHeader = bf.readInt();
        requestID = bf.readLong();
        reservedFlags = bf.readShort();
        reserved1 = bf.readInt();
        contentLength = bf.readInt();
        extensionParts = new ArrayList<>();
        parseRecursive(nextHeader, bf);
    }

    private void parseRecursive(int nextHeaderType, ByteBuf bf) {
        for (; nextHeaderType != 0; ) {
            ExtensionPart part = ExtensionPartFactory.parse(nextHeaderType, bf);
            extensionParts.add(part);
            nextHeaderType = part.nextType;
        }
    }

    public void appendPart(ExtensionPart part) {
        if (extensionParts == null) extensionParts = new ArrayList<>();
        extensionParts.add(part);
    }

    public String prettyPrint() {
        StringBuilder parts = new StringBuilder();
        if (extensionParts != null)
            for (ExtensionPart part : extensionParts)
                parts.append(part.prettyPrint()).append(" ");
        return String.format("major.minor:%d.%d %d %s", (int) majorVersion, (int) minorVersion, contentLength, parts.toString());
    }

    public <T extends ExtensionPart> List<T> getPart(Class<T> type) {
        List<T> ret = new ArrayList<>();
        if (extensionParts != null) {
            for (ExtensionPart part : extensionParts) {
                if (type.isAssignableFrom(part.getClass())) {
                    ret.add((T) part);
                }
            }
        }
        return ret;
    }
}
