/*
 *    Copyright (c) [2021] [Peking University]
 *    [BDWare DOIP SDK] is licensed under Mulan PSL v2.
 *    You can use this software according to the terms and conditions of the Mulan PSL v2.
 *    You may obtain a copy of Mulan PSL v2 at:
 *             http://license.coscl.org.cn/MulanPSL2
 *    THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 *    See the Mulan PSL v2 for more details.
 */

package org.bdware.doip.endpoint.client.v3;

import io.netty.channel.Channel;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.SimpleChannelInboundHandler;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.doip.codec.v3.DOIPV3Message;
import org.bdware.doip.codec.v3.TimeoutMsgFactory;
import org.bdware.doip.codec.v3.headers.ExceptionHead;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

@ChannelHandler.Sharable
public class ClientHandler extends SimpleChannelInboundHandler<DOIPV3Message> {
    static Logger logger = LogManager.getLogger(ClientHandler.class);
    public Channel channel;
    public TimeoutMsgFactory<DOIPV3Message> timeoutFactor = new TimeoutMsgFactory<DOIPV3Message>() {
        @Override
        public DOIPV3Message createTimoutMessage(long requestId, DOIPV3Message request, long timeout, String msg) {
            DOIPV3Message message = new DOIPV3Message();
            message.requestID = requestId;
            ExceptionHead exception = new ExceptionHead();
            exception.code = 2;
            exception.setContent(msg.getBytes(StandardCharsets.UTF_8));
            message.appendPart(exception);
            return message;
        }
    };
    ResponseWait sync = new ResponseWait<DOIPV3Message>(timeoutFactor);
    Random random = new Random();

    public void sendMessage(DOIPV3Message request, MessageCallback callback) {
        sendMessage(request, callback, 30);
    }

    public void sendMessage(DOIPV3Message request, MessageCallback callback, int timeoutSeconds) {
        if (callback == null) {
            logger.error("DoipMessageCallback is null, please check!");
            return;
        }
//        if (!channel.isWritable()) {
//            Thread.yield();
//            logger.info("network busy, yeild");
//        }
        int retryCount = 0;
        if (request.requestID == 0) {
            request.requestID = random.nextLong();
        }

        int MAX_RETRY_COUNT = 5;
        while (retryCount < MAX_RETRY_COUNT && !sync.waitResponse(request.requestID, callback, timeoutSeconds)) {
            request.requestID = random.nextLong();
            Thread.yield();
            retryCount++;
        }
        // logger.debug("writeAndFlush: " + new String(request.header.parameters.toByteArray()));
        // logger.debug("channel status: " + channel.isActive());

        if (retryCount >= MAX_RETRY_COUNT) {
            logger.error("waitObj.size() is too large! Could not get response: " + request.requestID);
            DOIPV3Message response = timeoutFactor.createTimoutMessage(request.requestID, request, 0, "waitObj.size too large!");
            callback.onResult(response);
        } else {
            channel.writeAndFlush(request);
        }
    }


    public DOIPV3Message sendMessageSync(DOIPV3Message request, int timeoutSeconds) {
        List<DOIPV3Message> ret = new ArrayList<>();
        final MessageCallback cb = new MessageCallback<DOIPV3Message>() {
            @Override
            public void onResult(DOIPV3Message msg) {
                ret.add(msg);
                synchronized (this) {
                    this.notify();
                }
            }
        };
        sendMessage(request, cb, timeoutSeconds);
        try {
            cb.wait(timeoutSeconds * 1000L);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        if (ret.size() > 0)
            return ret.get(0);
        DOIPV3Message response = timeoutFactor.createTimoutMessage(request.requestID, request, 0, "[NettyDoipClientHandler.sendMessageSync]");
        return response;
    }

    public void close() {
        channel.close();
    }

    public void setChannel(Channel c) {
        this.channel = c;
    }


    @Override
    protected void channelRead0(ChannelHandlerContext ctx, DOIPV3Message msg) {
        logger.debug("channelRead0 receive a message");
        sync.wakeUpAndRemove(msg.requestID, msg);
    }


}
