package org.bdware.irp3.codec.predefined;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import org.bdware.irp3.codec.ByteList;
import org.bdware.irp3.codec.Element;
import org.bdware.irp3.codec.UTF8String;
import java.util.ArrayList;
import java.util.List;

/**
 * The value of the element is a binary encoding of the public key which for key types considered
 * in this specification is as follows. First, there is a UTF8-string that describes the key type;
 * then, a two- byte option field reserved for future use; and finally, a key-type-dependent number
 * of length- prefixed byte-arrays that contains the public key itself. The key types in current use
 * are “DSA_PUB_KEY”, where there are four byte-arrays after the two-byte option field for the four
 * DSA parameters q, p, g, and y; and “RSA_PUB_KEY”, where after the two-byte option field are two
 * byte- arrays for the exponent and modulus, followed by an empty byte-array (four zero bytes).
 */
public class HS_Pubkey extends Element {
    public static String TYPE = "HS_PUBKEY";
    String keyType;
    short reserved;
    List<byte[]> publicKeyData; // Multiple byte arrays for key-type-dependent data

    public HS_Pubkey(Element element) {
        super(element);
        assert element.getType().equals(TYPE);
    }

    public HS_Pubkey() {
        setType(TYPE);
    }

    @Override
    public void fillFieldsFromValue() {
        ByteBuf byteBuf = Unpooled.wrappedBuffer(getValue());
        keyType = UTF8String.fromByteBuf(byteBuf);
        reserved = byteBuf.readShort();
        // Read multiple byte arrays for key-type-dependent data
        publicKeyData = new ArrayList<>();
        while (byteBuf.isReadable()) {
            byte[] data = ByteList.fromByteBuf(byteBuf);
            publicKeyData.add(data);
        }
        // Validate the parsed data
        validate();
    }

    @Override
    public byte[] calculateValue() {
        // Validate the public key data before encoding
        validate();
        
        ByteBuf byteBuf = Unpooled.directBuffer();
        UTF8String.toByteBuf(keyType, byteBuf);
        byteBuf.writeShort(reserved);
        // Write multiple byte arrays for key-type-dependent data
        for (byte[] data : publicKeyData) {
            ByteList.toByteBuf(data, byteBuf);
        }
        return ByteBufUtil.getBytes(byteBuf);
    }

    // Getter and setter methods
    public String getKeyType() {
        return keyType;
    }

    public void setKeyType(String keyType) {
        this.keyType = keyType;
    }

    public short getReserved() {
        return reserved;
    }

    public void setReserved(short reserved) {
        this.reserved = reserved;
    }

    public List<byte[]> getPublicKeyData() {
        return publicKeyData;
    }

    public void setPublicKeyData(List<byte[]> publicKeyData) {
        this.publicKeyData = publicKeyData;
    }

    /**
     * Validates the HSPubkey structure according to the protocol specification
     */
    public void validate() {
        if (keyType == null || keyType.isEmpty()) {
            throw new IllegalArgumentException("Key type cannot be empty");
        }

        if (publicKeyData == null) {
            throw new IllegalArgumentException("Public key data cannot be null");
        }

        switch (keyType) {
            case "DSA_PUB_KEY":
                // DSA requires exactly 4 byte arrays: q, p, g, y
                if (publicKeyData.size() != 4) {
                    throw new IllegalArgumentException(
                        String.format("DSA_PUB_KEY requires exactly 4 byte arrays (q, p, g, y), got %d", 
                                     publicKeyData.size()));
                }
                // Validate that all DSA parameters are non-empty
                for (int i = 0; i < publicKeyData.size(); i++) {
                    if (publicKeyData.get(i) == null || publicKeyData.get(i).length == 0) {
                        throw new IllegalArgumentException(
                            String.format("DSA parameter %d cannot be empty", i));
                    }
                }
                break;
            case "RSA_PUB_KEY":
                // RSA requires exactly 4 byte arrays: exponent, modulus, and two empty arrays for alignment
                if (publicKeyData.size() != 4) {
                    throw new IllegalArgumentException(
                        String.format("RSA_PUB_KEY requires exactly 4 byte arrays (exponent, modulus, empty, empty), got %d", 
                                     publicKeyData.size()));
                }
                // Validate exponent and modulus are non-empty
                if (publicKeyData.get(0) == null || publicKeyData.get(0).length == 0) {
                    throw new IllegalArgumentException("RSA exponent cannot be empty");
                }
                if (publicKeyData.get(1) == null || publicKeyData.get(1).length == 0) {
                    throw new IllegalArgumentException("RSA modulus cannot be empty");
                }
                // Third and fourth arrays must be empty (for alignment)
                if (publicKeyData.get(2) == null || publicKeyData.get(2).length != 0) {
                    throw new IllegalArgumentException(
                        String.format("RSA third byte array must be empty, got length %d", 
                                     publicKeyData.get(2) != null ? publicKeyData.get(2).length : -1));
                }
                if (publicKeyData.get(3) == null || publicKeyData.get(3).length != 0) {
                    throw new IllegalArgumentException(
                        String.format("RSA fourth byte array must be empty, got length %d", 
                                     publicKeyData.get(3) != null ? publicKeyData.get(3).length : -1));
                }
                break;
            default:
                throw new IllegalArgumentException("Unsupported key type: " + keyType);
        }
    }

}
