package org.bdware.irp3.client;

import io.netty.bootstrap.Bootstrap;
import io.netty.channel.*;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.LengthFieldBasedFrameDecoder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.irp.irplib.util.IrpCommon;
import org.bdware.irp3.codec.IrpMessage;
import org.bdware.irp3.codec.IrpMessageCodec;

import java.net.URI;
import java.net.URISyntaxException;

public class TcpClientChannel {
    final Bootstrap b = new Bootstrap();
    static EventLoopGroup group = new NioEventLoopGroup();
    private final ClientHandler handler;
    static Logger LOGGER = LogManager.getLogger(TcpClientChannel.class);
    private static final int DefaultTimeout = 5;

    public TcpClientChannel() {
        b.option(ChannelOption.CONNECT_TIMEOUT_MILLIS, 5000);
        b.option(ChannelOption.SO_KEEPALIVE, true);
        b.group(group);
        handler = new ClientHandler();
        b.channel(NioSocketChannel.class)
                .option(ChannelOption.TCP_NODELAY, true)
                .handler(
                        new ChannelInitializer<SocketChannel>() {
                            @Override
                            protected void initChannel(SocketChannel ch) {
                                ChannelPipeline p = ch.pipeline();
                                p.addLast(new LengthFieldBasedFrameDecoder(IrpCommon.MAX_MESSAGE_PACKET_LENGTH, 16, 4, 0, 0))
                                        .addLast(new IrpMessageCodec())
                                        .addLast(handler);
                            }
                        });
    }

    public void sendMessage(IrpMessage message, ResultCallback<IrpMessage> cb) {
        handler.sendMessage(message, cb, 5);
    }

    static class SyncCallback implements ResultCallback<IrpMessage> {
        private IrpMessage response;

        @Override
        public synchronized void onResult(IrpMessage message) {
            response = message;
            this.notifyAll();
        }
    }

    public IrpMessage sendMessageSync(IrpMessage msg) {
        return sendMessageSync(msg, DefaultTimeout);
    }

    public IrpMessage sendMessageSync(IrpMessage msg, int timeoutInSeconds) {
        SyncCallback rcb = new SyncCallback();
        handler.sendMessage(msg, rcb, timeoutInSeconds);
        synchronized (rcb) {
            try {
                rcb.wait(timeoutInSeconds * 1100);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        if (rcb.response == null)
            return null;
        return rcb.response;
    }

    public void connect(String targetUrl) throws URISyntaxException {
        URI uri = new URI(targetUrl);
        LOGGER.info("[URI Parse]scheme:" + uri.getScheme() + "  host: " + uri.getHost() + "  port: " + uri.getPort());
        try {
            Channel channel = b.connect(uri.getHost(), uri.getPort()).sync().channel();
            handler.setChannel(channel);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
