package org.bdware.irp3.handler;

import io.netty.channel.ChannelHandlerContext;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bdware.irp3.body.ErrorResponse;
import org.bdware.irp3.codec.MessageBody;
import org.bdware.irp3.server.RequestHandler;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public abstract class ReflectiveRequestHandler implements RequestHandler {
    static Logger logger = LogManager.getLogger(ReflectiveRequestHandler.class);

    static class Pair {
        IrpHandlerBase obj;
        Method method;
    }

    Map<Class, Pair> handlerMap;
    static Logger LOGGER = LogManager.getLogger(ReflectiveRequestHandler.class);
    List<IrpHandlerBase> handlers;

    public ReflectiveRequestHandler() {
        handlerMap = new ConcurrentHashMap<>();
        handlers = new ArrayList<>();
    }

    public void addHandler(IrpHandlerBase handler) {

        Class handlerClass = handler.getClass();
        for (; !handlerClass.equals(Object.class); ) {
            putHandlerMethod(handler, handlerClass);
            Class[] interfaces = handlerClass.getInterfaces();
            for (Class clz : interfaces) {
                putHandlerMethod(handler, handlerClass);
            }
            handlerClass = handlerClass.getSuperclass();
        }
    }

    private void putHandlerMethod(IrpHandlerBase handler, Class handlerClass) {
        Method[] methods = handlerClass.getDeclaredMethods();
        for (Method m : methods) {
            IrpProcessor a = m.getAnnotation(IrpProcessor.class);
            if (a != null) {
                Class<?>[] parameters = m.getParameterTypes();
                if (parameters.length != 2) {
                    if (!parameters[0].equals(ChannelHandlerContext.class) || !MessageBody.class.isAssignableFrom(parameters[1])) {
                        LOGGER.error("IrpProcessor parameter error:" + handlerClass.getCanonicalName() + ":" + m.getName());
                        continue;
                    }
                }
                if (!m.getReturnType().equals(Void.TYPE) && !MessageBody.class.isAssignableFrom(m.getReturnType())) {
                    LOGGER.error("IrpProcessor return type error:" + handlerClass.getCanonicalName() + ":" + m.getName());
                    continue;
                }
                putHandler(handler, parameters[1], m);
            }
        }
    }

    private void putHandler(IrpHandlerBase handler, Class<?> parameter, Method m) {
        m.setAccessible(true);
        Pair p = new Pair();
        p.obj = handler;
        p.method = m;
        handlerMap.put(parameter, p);
    }

    @Override
    public MessageBody onRequest(ChannelHandlerContext ctx, MessageBody msg) {
        Pair pair = handlerMap.get(msg.getClass());
        if (pair != null) {
            try {
                return (MessageBody) pair.method.invoke(pair.obj, ctx, msg);
            } catch (Exception e) {
                e.printStackTrace();
                return new ErrorResponse(e.getMessage());
            }
        } else return new ErrorResponse("unsupported type:" + msg.getClass().getCanonicalName());
    }
}
